import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Dialog, DialogContent } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { ScrollArea } from '@/components/ui/scroll-area';
import { useGetReverseDNS, useGetWhois } from '@/components/suricata/data/queries';
import { Skeleton } from '@/components/ui/skeleton';
import { useTranslation } from 'react-i18next';

type DataActionDialogProps = {
  ip: 'src' | 'dest';
  address: string;
  open: boolean;
  onOpenChange: (open: boolean) => void;
};

const whoisNotInstalled =
  'Whois not downloaded please install whois via your package manager.\n\nFor Debian/Ubuntu: sudo apt install whois\nFor RHEL/CentOS/Oracle: sudo yum install whois';

export function WhoisDialog({ ip, address, open, onOpenChange }: DataActionDialogProps) {
  const { t } = useTranslation();
  const { data: whoisText, isLoading } = useGetWhois(address);
  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="border-none bg-transparent p-0 [&>button]:hidden">
        <Card className="w-150 gap-2">
          <CardHeader>
            <CardTitle>
              {t('Whois for ')} {address}
            </CardTitle>
            <CardDescription>
              {ip === 'src'
                ? t('Whois information for the source IP address')
                : t('Whois information for the destination IP address')}
            </CardDescription>
          </CardHeader>
          <CardContent className="bg-sidebar-ring/30 border-border/50 m-2 rounded border">
            <ScrollArea className="-mr-5 sm:h-100 lg:h-125 xl:h-150">
              {isLoading ? (
                <div className="space-y-2 p-4">
                  {[...Array(30)].map((_, i) => (
                    <Skeleton key={i} className="h-4 w-5/6" />
                  ))}
                </div>
              ) : (
                <pre className="py-2 font-mono text-sm whitespace-pre-wrap">
                  <span className="select-text hover:cursor-text">{whoisText?.whois ?? whoisNotInstalled}</span>
                </pre>
              )}
            </ScrollArea>
          </CardContent>
          <CardFooter>
            <Button onClick={() => onOpenChange(false)}>{t('Close')}</Button>
          </CardFooter>
        </Card>
      </DialogContent>
    </Dialog>
  );
}

export function RDNSDialog({ ip, address, open, onOpenChange }: DataActionDialogProps) {
  const { t } = useTranslation();
  const { data: rdnsData, isLoading } = useGetReverseDNS(address);
  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="border-none bg-transparent p-0 [&>button]:hidden">
        <Card className="w-150 gap-2">
          <CardHeader>
            <CardTitle>
              {t('Reverse DNS for ')} {address}
            </CardTitle>
            <CardDescription>
              {ip === 'src'
                ? t('Reverse DNS information for the source IP address')
                : t('Reverse DNS information for the destination IP address')}
            </CardDescription>
          </CardHeader>
          <CardContent className="bg-sidebar-ring/30 border-border/50 m-2 rounded border">
            {isLoading ? (
              <div className="py-2.5">
                <Skeleton className="h-4 w-5/6" />
              </div>
            ) : (
              <pre className="py-2 font-mono text-sm whitespace-pre-wrap">
                <span className="select-text hover:cursor-text">{rdnsData?.ptr ?? 'No PTR record found'}</span>
              </pre>
            )}
          </CardContent>
          <CardFooter>
            <Button onClick={() => onOpenChange(false)}>{t('Close')}</Button>
          </CardFooter>
        </Card>
      </DialogContent>
    </Dialog>
  );
}
