import { useState } from 'react';
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { useTranslation } from 'react-i18next';
import { Ellipsis, Radar, ShieldQuestion, Waypoints } from 'lucide-react';
import { WhoisDialog, RDNSDialog } from '@/components/suricata/data/DataActionDialogs';
import AddNmapScanDialog from '@/components/nmap/scans/AddNmapScanDialog';
import SharkFin from '@/components/ui/icons/SharkFin';
import SearchRingBufferDialog from '@/components/wireshark/ringbuffer/SearchRingBufferDialog';
import { useGetIntegrations } from '@/components/queries';

export function DataActionDropdown({ src, dest }: { src: string; dest: string }) {
  const { t } = useTranslation();
  const [action, setAction] = useState<'whois' | 'rdns' | null>(null);
  const [addNmapScanDialogOpen, setAddNmapScanDialogOpen] = useState(false);
  const [searchDialogOpen, setSearchDialogOpen] = useState(false);
  const [ip, setIP] = useState<'src' | 'dest'>('src');
  const { data: integrationStatus } = useGetIntegrations();

  return (
    <>
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost" size="sm" className="ml-auto flex h-8">
            <Ellipsis />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="end">
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={(e) => {
              e.stopPropagation();
              setIP('src');
              setAction('whois');
            }}
          >
            <ShieldQuestion />
            {t('Whois Src IP')}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={(e) => {
              e.stopPropagation();
              setIP('dest');
              setAction('whois');
            }}
          >
            <ShieldQuestion />
            {t('Whois Dest IP')}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={(e) => {
              e.stopPropagation();
              setIP('src');
              setAction('rdns');
            }}
          >
            <Waypoints />
            {t('Reverse DNS Src IP')}
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer"
            onClick={(e) => {
              e.stopPropagation();
              setIP('dest');
              setAction('rdns');
            }}
          >
            <Waypoints />
            {t('Reverse DNS Dest IP')}
          </DropdownMenuItem>
          {!!integrationStatus?.nmap && (
            <DropdownMenuItem
              className="cursor-pointer"
              onClick={(e) => {
                e.stopPropagation();
                setIP('src');
                setAddNmapScanDialogOpen(true);
              }}
            >
              <Radar />
              {t('Nmap Scan Src IP')}
            </DropdownMenuItem>
          )}
          {!!integrationStatus?.nmap && (
            <DropdownMenuItem
              className="cursor-pointer"
              onClick={(e) => {
                e.stopPropagation();
                setIP('dest');
                setAddNmapScanDialogOpen(true);
              }}
            >
              <Radar />
              {t('Nmap Scan Dest IP')}
            </DropdownMenuItem>
          )}
          {!!integrationStatus?.wireshark && integrationStatus?.ringbuffer && (
            <DropdownMenuItem
              className="cursor-pointer"
              onClick={(e) => {
                e.stopPropagation();
                setSearchDialogOpen(true);
              }}
            >
              <SharkFin />
              {t('Search Wireshark')}
            </DropdownMenuItem>
          )}
        </DropdownMenuContent>
      </DropdownMenu>
      <div onClick={(e) => e.stopPropagation()}>
        <AddNmapScanDialog
          open={addNmapScanDialogOpen}
          onOpenChange={setAddNmapScanDialogOpen}
          parameters={ip === 'src' ? src : dest}
        >
          <span />
        </AddNmapScanDialog>
        <SearchRingBufferDialog open={searchDialogOpen} onOpenChange={setSearchDialogOpen} src={src} dst={dest} />
      </div>
      <div onClick={(e) => e.stopPropagation()}>
        <WhoisDialog
          ip={ip}
          address={ip === 'src' ? src : dest}
          open={action === 'whois'}
          onOpenChange={(open) => !open && setAction(null)}
        />
        <RDNSDialog
          ip={ip}
          address={ip === 'src' ? src : dest}
          open={action === 'rdns'}
          onOpenChange={(open) => !open && setAction(null)}
        />
      </div>
    </>
  );
}
