import axios, { AxiosError } from '@/lib/axios';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { toast } from 'sonner';
import { SuricataData, WhoisData, ReverseDNSData, ImportPcapParams } from '@/components/suricata/data/types';
import { useTranslation } from 'react-i18next';

export function useGetSuricataData(per_page: number, page: number, sortBy: string, sortOrder: string, filter: string) {
  return useQuery<SuricataData>({
    queryKey: ['AllSuricataData', per_page, page, sortBy, sortOrder, filter],
    queryFn: async () => {
      const response = await axios.get<SuricataData>('/api/v1/suricata/data', {
        params: {
          page,
          per_page,
          sortBy,
          sortOrder,
          filter,
        },
      });
      return response.data;
    },
  });
}

export const fetchFullJSON = async (id: number) => {
  const res = await axios.get<Record<string, unknown>>(`/api/v1/suricata/data/full-log-json?id=${id}`);
  return res.data;
};

export function useGetFullJSONMutation() {
  return useMutation({
    mutationFn: ({ id }: { id: number }) => fetchFullJSON(id),
  });
}

export function useGetWhois(ip: string) {
  return useQuery<WhoisData>({
    queryKey: ['whois', ip],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/system/whois?ip=${ip}`);
      return response.data;
    },
    enabled: !!ip,
    retry: 1,
  });
}

export function useGetJsonFiles() {
  return useQuery<string[]>({
    queryKey: ['JsonFiles'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/suricata/data/json-files');
      return response.data;
    },
  });
}

export function useGetReverseDNS(ip: string) {
  return useQuery<ReverseDNSData>({
    queryKey: ['reverseDNS', ip],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/system/reverse-dns?ip=${ip}`);
      return response.data;
    },
    enabled: !!ip,
  });
}

export function useImportPcap() {
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (data: ImportPcapParams) => {
      const formData = new FormData();
      formData.append('file', data.file as File);
      formData.append('output_file_name', data.output_file_name);
      return axios.post('/api/v1/suricata/data/import-pcap', formData, {
        headers: { 'Content-Type': 'multipart/form-data' },
      });
    },
    onMutate: () => {
      toast(t('Scanning PCAP file (this may take a moment)...'));
    },
    onError: () => {
      toast.error(t('Failed to import Pcap file'));
    },
    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * attemptIndex, 5000),
  });
}

export function useDeleteSuricataData() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async ({ jsonFile }: { jsonFile: string }) => {
      return axios.delete(`/api/v1/suricata/data/${encodeURIComponent(jsonFile)}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['AllSuricataData'] });
      queryClient.invalidateQueries({ queryKey: ['JsonFiles'] });
      queryClient.invalidateQueries({ queryKey: ['suricataConfig'] });
      toast.success(t('Suricata data deleted successfully'));
    },
    onError: (error) => {
      const err = error as AxiosError<{ error: string }>;
      toast.error(err.response?.data?.error ?? 'Something went wrong');
    },
  });
}
