import { Pie, PieChart, Cell } from 'recharts';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from '@/components/ui/chart';
import { useGetSeverities } from '@/components/suricata/alerts/queries';
import React from 'react';
import { useTranslation } from 'react-i18next';
import { cn } from '@/lib/utils';

const red = 'oklch(0.645 0.246 16.439)';
const orange = 'oklch(0.769 0.188 70.08)';
const green = 'oklch(0.696 0.17 162.48)';

function getSeverityColor(severity: number): string {
  switch (severity) {
    case 1:
      return red;
    case 2:
      return orange;
    case 3:
      return green;
    default:
      return '#1e88e5';
  }
}

export function SeverityPieChart({ className }: { className?: string }) {
  const { t } = useTranslation();
  const title = t('Severity Distribution From The Week');
  const { data: severityData } = useGetSeverities();
  const chartData = React.useMemo(() => {
    if (!severityData) return [];
    return severityData.map((item) => ({
      label: `Severity ${item.severity}`,
      value: Number(item.count),
      fill: getSeverityColor(Number(item.severity)),
    }));
  }, [severityData]);

  const chartConfig: ChartConfig = React.useMemo(() => {
    if (!chartData.length) {
      return {
        value: { label: 'Value' },
      } satisfies ChartConfig;
    }

    return {
      value: { label: 'Value' },
      ...Object.fromEntries(chartData.map((item) => [item.label, { label: item.label, color: item.fill }])),
    } satisfies ChartConfig;
  }, [chartData]);

  return (
    <Card className={cn('h-full', className)}>
      <CardHeader className="items-center pb-2">
        <CardTitle>{title}</CardTitle>
      </CardHeader>
      <CardContent className="flex h-full w-full items-center justify-center">
        {chartData.length === 0 ? (
          <div className="text-muted-foreground text-center">{t('No severity data available')}</div>
        ) : (
          <ChartContainer config={chartConfig} className="h-full w-full">
            <PieChart>
              <ChartTooltip cursor={false} content={<ChartTooltipContent className="w-40" hideLabel />} />
              <Pie data={chartData} dataKey="value" nameKey="label" outerRadius="80%">
                {chartData.map((item) => (
                  <Cell key={item.label} fill={item.fill} />
                ))}
              </Pie>
              <ChartLegend className="text-nowrap" content={<ChartLegendContent />} />
            </PieChart>
          </ChartContainer>
        )}
      </CardContent>
    </Card>
  );
}
