import { Skeleton } from '@/components/ui/skeleton';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { useTranslation } from 'react-i18next';
import { useGetTopAlerts } from '@/components/suricata/alerts/queries';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { useEffect, useRef, useState } from 'react';
import { cn } from '@/lib/utils';

export function TopAlertsTable({ className }: { className?: string }) {
  const { data: alerts, isLoading } = useGetTopAlerts();

  const { t } = useTranslation();
  return (
    <Card className={cn('h-full', className)}>
      <CardHeader>
        <CardTitle>{t('Top Suricata Alerts From This Week')}</CardTitle>
      </CardHeader>
      <CardContent>
        <Table className="h-full">
          <TableHeader>
            <TableRow>
              <TableHead>{t('Alert Signature')}</TableHead>
              <TableHead className="text-right">{t('Count')}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {isLoading ? (
              Array.from({ length: 8 }).map((_, idx) => (
                <TableRow key={idx}>
                  <TableCell colSpan={2}>
                    <Skeleton className="h-5 w-full" />
                  </TableCell>
                </TableRow>
              ))
            ) : alerts && alerts.length > 0 ? (
              alerts.map((alert, idx) => (
                <TableRow key={idx}>
                  <TruncatedCell text={alert.signature_name} />
                  <TableCell className="text-right font-medium">{alert.count}</TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={2} className="text-muted-foreground text-center">
                  {t('No alert data available')}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </CardContent>
    </Card>
  );
}

function TruncatedCell({ text }: { text: string }) {
  const [isTruncated, setIsTruncated] = useState(false);
  const spanRef = useRef<HTMLSpanElement>(null);

  useEffect(() => {
    const el = spanRef.current;
    if (el) {
      setIsTruncated(el.scrollWidth > el.clientWidth);
    }
  }, [text]);

  const content = (
    <span
      ref={spanRef}
      className="block cursor-default truncate sm:max-w-[25vw] md:max-w-[25vw] lg:max-w-[12vw] [@media(min-width:1800px)]:max-w-[15vw]"
    >
      {text}
    </span>
  );

  return (
    <TableCell>
      {isTruncated ? (
        <TooltipProvider>
          <Tooltip>
            <TooltipTrigger asChild>{content}</TooltipTrigger>
            <TooltipContent side="top">{text}</TooltipContent>
          </Tooltip>
        </TooltipProvider>
      ) : (
        content
      )}
    </TableCell>
  );
}
