import { Sheet, SheetFooter, SheetContent, SheetHeader, SheetTitle, SheetDescription } from '@/components/ui/sheet';
import { Form, FormLabel, FormControl, FormField, FormItem, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { useForm, SubmitHandler, Controller } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useTranslation } from 'react-i18next';
import { Select, SelectItem, SelectTrigger, SelectValue, SelectContent } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Button } from '@/components/ui/button';
import { useEditRule, useGetFullRule } from '@/components/suricata/rules/queries';
import { useEffect, useState } from 'react';
import { Trash } from 'lucide-react';
import { ScrollArea } from '@/components/ui/scroll-area';

type AddEditRuleSheetProps = {
  open: boolean;
  setOpen: (open: boolean) => void;
  sid: string;
  title: string;
  description: string;
};

const validateAddressAndPort = (
  data: Record<'source ip' | 'source port' | 'destination ip' | 'destination port', string>,
  ctx: z.RefinementCtx,
  label: 'Source' | 'Destination',
  ipKey: 'source ip' | 'destination ip',
  portKey: 'source port' | 'destination port'
) => {
  const missingIp = !data[ipKey]?.trim();
  const missingPort = !data[portKey]?.trim();

  if (!missingIp && !missingPort) {
    return;
  }

  if (missingIp && missingPort) {
    const message = `${label} address and port are required`;
    ctx.addIssue({
      code: 'custom',
      path: [ipKey],
      message,
    });
    return;
  }

  if (missingIp) {
    ctx.addIssue({
      code: 'custom',
      path: [ipKey],
      message: `${label} address is required`,
    });
  }

  if (missingPort) {
    ctx.addIssue({
      code: 'custom',
      path: [portKey],
      message: `${label} port is required`,
    });
  }
};

const coreSuricataRuleSchema = z.object({
  action: z.enum(['alert', 'drop', 'reject', 'pass', 'rejectsrc', 'rejectdst', 'rejectboth'], {
    error: 'Action is required',
  }),
  protocol: z.string().min(1, 'Protocol is required').regex(/^\S+$/, 'Protocol cannot contain spaces'),
  direction: z.enum(['outbound', 'bidirectional'], {
    error: 'Direction is required',
  }),
  msg: z.string().min(1, 'Message is required'),
  sid: z
    .string()
    .min(1, 'SID is required')
    .regex(/^\S+$/, 'SID cannot contain spaces')
    .regex(/^\d+$/, 'SID must be a postive number')
    .refine((val) => parseInt(val) > 0, {
      message: 'SID must be greater than 0',
    }),
  options: z.record(z.string(), z.union([z.string().min(1, 'Option value is required'), z.boolean()])).optional(),
  status: z.enum(['Active', 'Inactive']).optional(),
});

const addressPortSchema = z
  .object({
    'source ip': z
      .string()
      .min(1, 'Source IP is required')
      .regex(/^[^\s]+$/, 'Source IP cannot contain spaces'),
    'source port': z
      .string()
      .min(1, 'Source port is required')
      .regex(/^[^\s]+$/, 'Source port cannot contain spaces'),
    'destination ip': z
      .string()
      .min(1, 'Destination IP is required')
      .regex(/^[^\s]+$/, 'Destination IP cannot contain spaces'),
    'destination port': z
      .string()
      .min(1, 'Destination port is required')
      .regex(/^[^\s]+$/, 'Destination port cannot contain spaces'),
  })
  .superRefine((data, ctx) => {
    validateAddressAndPort(data, ctx, 'Source', 'source ip', 'source port');
    validateAddressAndPort(data, ctx, 'Destination', 'destination ip', 'destination port');
  });

const suricataRuleSchema = coreSuricataRuleSchema.and(addressPortSchema);

type SuricataRuleFormData = z.infer<typeof suricataRuleSchema>;

export function AddEditRuleSheet({ open, setOpen, sid, title, description }: AddEditRuleSheetProps) {
  const { t } = useTranslation();
  const editMutation = useEditRule();
  const { data } = useGetFullRule(sid, { enabled: open && !!sid && sid !== '-1' });
  const [addingOption, setAddingOption] = useState(false);
  const [newOptionKey, setNewOptionKey] = useState('');
  const [newOptionType, setNewOptionType] = useState<'string' | 'boolean'>('string');

  const form = useForm<SuricataRuleFormData>({
    resolver: zodResolver(suricataRuleSchema),
    defaultValues: {
      action: undefined,
      protocol: '',
      'source ip': '',
      'source port': '',
      direction: undefined,
      'destination ip': '',
      'destination port': '',
      msg: '',
      sid: undefined,
      status: 'Active',
      options: {},
    },
  });

  // for editing rules and resetting for new rules
  useEffect(() => {
    if (!open) return;

    if (sid === '-1') {
      form.reset({
        action: undefined,
        protocol: '',
        'source ip': '',
        'source port': '',
        direction: undefined,
        'destination ip': '',
        'destination port': '',
        msg: '',
        sid: '',
        status: 'Active',
        options: {},
      });
    } else if (data) {
      // Load existing rule data for editing
      const currentOptions = data.options ?? {};

      // Extract msg and sid from options
      const msgValue = typeof currentOptions.msg === 'string' ? currentOptions.msg : '';
      const sidValue =
        typeof currentOptions.sid === 'string'
          ? currentOptions.sid
          : typeof currentOptions.sid === 'number'
            ? String(currentOptions.sid)
            : '';

      // Filter options to exclude msg and sid, only include string or boolean values
      const validOptions: Record<string, string | boolean> = {};
      Object.entries(currentOptions).forEach(([key, value]) => {
        if (key !== 'msg' && key !== 'sid' && (typeof value === 'string' || typeof value === 'boolean')) {
          validOptions[key] = value;
        }
      });

      form.reset({
        action:
          (data.action as 'alert' | 'drop' | 'reject' | 'pass' | 'rejectsrc' | 'rejectdst' | 'rejectboth') ?? 'alert',
        protocol: data.protocol ?? '',
        'source ip': data['source ip'] ?? '',
        'source port': data['source port'] ?? '',
        direction: (data.direction === 'outbound' || data.direction === 'bidirectional'
          ? data.direction
          : 'outbound') as 'outbound' | 'bidirectional',
        'destination ip': data['destination ip'] ?? '',
        'destination port': data['destination port'] ?? '',
        msg: msgValue,
        sid: sidValue,
        status: (data.status === 'Active' || data.status === 'Inactive' ? data.status : 'Active') as
          | 'Active'
          | 'Inactive',
        options: validOptions,
      });
    }
  }, [open, data, form, sid]);

  const onSubmit: SubmitHandler<SuricataRuleFormData> = (formData) => {
    // Merge msg and sid into options as the backend expects them there
    const { msg, sid: formSid, options, ...rest } = formData;

    const submissionData = {
      ...rest,
      options: {
        ...options,
        msg,
        sid: formSid,
      },
    };

    editMutation.mutate({
      original_sid: String(sid),
      new_rule: submissionData,
    });

    form.reset();
    setAddingOption(false);
    setNewOptionKey('');
    setNewOptionType('string');
    setOpen(false);
  };

  const removeOption = (key: string) => {
    const currentOptions = form.getValues('options') ?? {};
    const { [key]: removed, ...rest } = currentOptions;
    form.setValue('options', rest);
  };

  const handleOpenChange = (isOpen: boolean) => {
    if (isOpen) {
      setOpen(true);
      return;
    }

    setOpen(false);
    form.reset();
    setAddingOption(false);
    setNewOptionKey('');
    setNewOptionType('string');
  };

  const sourceErrorMessage =
    form.formState.errors['source ip']?.message || form.formState.errors['source port']?.message;

  const destinationErrorMessage =
    form.formState.errors['destination ip']?.message || form.formState.errors['destination port']?.message;

  return (
    <Sheet open={open} onOpenChange={handleOpenChange}>
      <SheetContent className="flex flex-col sm:max-w-[600px]">
        <SheetHeader className="mb-4">
          <SheetTitle>{title}</SheetTitle>
          <SheetDescription>{description}</SheetDescription>
        </SheetHeader>
        <ScrollArea className="overflow-y-hidden pr-4">
          <Form {...form}>
            <form id="edit-rule" onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-4">
              <FormField
                control={form.control}
                name="action"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 space-y-0 pr-1">
                    <FormLabel className="text-right">{t('Action')}</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select a rule action')} />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="alert">{t('alert')}</SelectItem>
                        <SelectItem value="pass">{t('pass')}</SelectItem>
                        <SelectItem value="drop">{t('drop')}</SelectItem>
                        <SelectItem value="reject">{t('reject')}</SelectItem>
                        <SelectItem value="rejectsrc">{t('rejectsrc')}</SelectItem>
                        <SelectItem value="rejectdst">{t('rejectdst')}</SelectItem>
                        <SelectItem value="rejectboth">{t('rejectboth')}</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage className="col-span-3 col-start-2" />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="protocol"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 space-y-0 pr-1">
                    <FormLabel className="text-right">{t('Protocol')}</FormLabel>
                    <FormControl>
                      <Input {...field} className="col-span-3" />
                    </FormControl>
                    <FormMessage className="col-span-3 col-start-2" />
                  </FormItem>
                )}
              />
              <div className="grid grid-cols-4">
                <FormLabel className={`text-right ${sourceErrorMessage ? 'text-destructive' : ''}`}>
                  {t('Source')}
                </FormLabel>
                <div className="col-span-3 flex flex-col gap-2">
                  <div className="flex gap-4">
                    <FormField
                      control={form.control}
                      name="source ip"
                      render={({ field }) => (
                        <FormItem className="flex-1 space-y-0">
                          <FormControl>
                            <Input {...field} placeholder={t('IP Address')} className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                    <span className="-mt-1 flex items-center">:</span>
                    <FormField
                      control={form.control}
                      name="source port"
                      render={({ field }) => (
                        <FormItem className="w-18 space-y-0 pr-1">
                          <FormControl>
                            <Input {...field} placeholder={t('Port')} className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </div>
                  {sourceErrorMessage && <p className="text-destructive text-sm">{sourceErrorMessage}</p>}
                </div>
              </div>
              <div className="grid grid-cols-4">
                <FormLabel className={`text-right ${destinationErrorMessage ? 'text-destructive' : ''}`}>
                  {t('Destination')}
                </FormLabel>
                <div className="col-span-3 flex flex-col gap-2">
                  <div className="flex gap-4">
                    <FormField
                      control={form.control}
                      name="destination ip"
                      render={({ field }) => (
                        <FormItem className="flex-1 space-y-0">
                          <FormControl>
                            <Input {...field} placeholder={t('IP Address')} className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                    <span className="-mt-1 flex items-center">:</span>
                    <FormField
                      control={form.control}
                      name="destination port"
                      render={({ field }) => (
                        <FormItem className="w-18 space-y-0 pr-1">
                          <FormControl>
                            <Input {...field} placeholder={t('Port')} className="w-full" />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </div>
                  {destinationErrorMessage && <p className="text-destructive text-sm">{destinationErrorMessage}</p>}
                </div>
              </div>
              <FormField
                control={form.control}
                name="direction"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 pr-1">
                    <FormLabel className="text-right text-nowrap">{t('Direction')}</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select a network direction')} />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="outbound">{t('Outbound (->)')}</SelectItem>
                        <SelectItem value="bidirectional">{t('Bidirectional (<>)')}</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage className="col-span-3 col-start-2" />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="msg"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 space-y-0 pr-1">
                    <FormLabel className="text-right">{t('Message')}</FormLabel>
                    <FormControl>
                      <Input {...field} className="col-span-3" placeholder={t('Rule description')} />
                    </FormControl>
                    <FormMessage className="col-span-3 col-start-2" />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="sid"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-4 space-y-0 pr-1">
                    <FormLabel className="text-right">{t('SID')}</FormLabel>
                    <FormControl>
                      <Input {...field} className="col-span-3" placeholder="1000000" value={field.value ?? ''} />
                    </FormControl>
                    <FormMessage className="col-span-3 col-start-2" />
                  </FormItem>
                )}
              />
              <div className="py-5">{t('Rule Options')}</div>
              {Object.entries(form.watch('options') ?? {})
                .filter(([key]) => key !== 'msg' && key !== 'sid')
                .map(([key, value]) =>
                  typeof value === 'string' ? (
                    <Controller
                      key={key}
                      name={`options.${key}`}
                      control={form.control}
                      defaultValue={value}
                      render={({ field, fieldState }) => (
                        <FormItem className="flex flex-col space-y-1 pr-1">
                          <div className="flex space-y-0">
                            <FormLabel className="min-w-[120px]">{t(key)}</FormLabel>
                            <FormControl>
                              <Input
                                {...field}
                                value={typeof field.value === 'string' ? field.value : String(field.value ?? '')}
                                className="w-full"
                              />
                            </FormControl>
                            <Button
                              type="button"
                              variant="ghost"
                              size="sm"
                              onClick={() => removeOption(key)}
                              title={t('Remove')}
                            >
                              <Trash />
                            </Button>
                          </div>
                          {fieldState.error && (
                            <FormMessage className="ml-[120px]">{fieldState.error.message}</FormMessage>
                          )}
                        </FormItem>
                      )}
                    />
                  ) : null
                )}
              {Object.entries(form.watch('options') ?? {})
                .filter(([key]) => key !== 'msg' && key !== 'sid')
                .map(([key, value]) =>
                  typeof value === 'boolean' ? (
                    <Controller
                      key={key}
                      name={`options.${key}`}
                      control={form.control}
                      defaultValue={value}
                      render={() => (
                        <FormItem className="grid grid-cols-6 items-center gap-4 space-y-0 pr-1">
                          <FormLabel className="text-right">{t(key)}</FormLabel>
                          <FormControl>
                            <div className="flex items-center gap-2">
                              <Button
                                type="button"
                                variant="ghost"
                                size="sm"
                                onClick={() => removeOption(key)}
                                className="ml-2"
                                title={t('Remove')}
                              >
                                <Trash />
                              </Button>
                            </div>
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  ) : null
                )}
              <FormField
                control={form.control}
                name="status"
                render={({ field }) => (
                  <FormItem className="grid grid-cols-6 space-y-0 pt-2 pr-1">
                    <FormLabel className="text-right">{t('Enabled')}</FormLabel>
                    <FormControl>
                      <Checkbox
                        checked={field.value === 'Active'}
                        onCheckedChange={(checked) => field.onChange(checked ? 'Active' : 'Inactive')}
                      />
                    </FormControl>
                  </FormItem>
                )}
              />
            </form>
          </Form>
        </ScrollArea>
        <SheetFooter className="flex flex-col gap-4 sm:flex-row sm:items-end sm:justify-between">
          <div className="flex flex-wrap gap-2">
            {!addingOption ? (
              <Button type="button" variant="outline" size="sm" onClick={() => setAddingOption(true)}>
                +
              </Button>
            ) : (
              <>
                <Input
                  value={newOptionKey}
                  onChange={(e) => setNewOptionKey(e.target.value)}
                  placeholder={t('Option name')}
                  className="w-50"
                  autoFocus
                />
                <Select value={newOptionType} onValueChange={(v) => setNewOptionType(v as 'string' | 'boolean')}>
                  <SelectTrigger className="w-40">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="string">{t('With Argument')}</SelectItem>
                    <SelectItem value="boolean">{t('Without Argument')}</SelectItem>
                  </SelectContent>
                </Select>
                <div className="justify-end">
                  <Button
                    type="button"
                    size="sm"
                    onClick={() => {
                      if (!newOptionKey) return;
                      form.setValue(`options.${newOptionKey}`, newOptionType === 'boolean' ? true : '');
                      setAddingOption(false);
                      setNewOptionKey('');
                      setNewOptionType('string');
                    }}
                    variant="default"
                  >
                    {t('Add')}
                  </Button>
                  <Button
                    type="button"
                    size="sm"
                    variant="ghost"
                    onClick={() => {
                      setAddingOption(false);
                      setNewOptionKey('');
                      setNewOptionType('string');
                    }}
                  >
                    {t('Cancel')}
                  </Button>
                </div>
              </>
            )}
          </div>
          <Button form="edit-rule" type="submit" className="w-full self-end sm:w-auto">
            {t('Submit')}
          </Button>
        </SheetFooter>
      </SheetContent>
    </Sheet>
  );
}
