import { ColumnDef } from '@tanstack/react-table';
import { Tooltip, TooltipContent, TooltipTrigger, TooltipProvider } from '@/components/ui/tooltip';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { SuricataRules } from '@/components/suricata/rules/types';
import { useState, useRef, useEffect } from 'react';
import { RulesDropdown } from '@/components/suricata/rules/RulesDropdown';
import { Checkbox } from '@/components/ui/checkbox';
import { Switch } from '@/components/ui/switch';
import { hasPermission } from '@/components/role/utils';
import { Role } from '@/components/role/types';

function TruncatedTextCell({ value }: { value: string }) {
  const textRef = useRef<HTMLDivElement>(null);
  const [isOverflowing, setIsOverflowing] = useState(false);

  useEffect(() => {
    const checkOverflow = () => {
      const el = textRef.current;
      if (el) {
        const isOver = el.scrollWidth > el.clientWidth;
        setIsOverflowing(isOver);
      }
    };

    let resizeTimeout: NodeJS.Timeout;
    const handleResize = () => {
      clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(() => {
        checkOverflow();
      }, 100);
    };

    checkOverflow();
    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
      clearTimeout(resizeTimeout);
    };
  }, [value]);

  const TextContent = (
    <div ref={textRef} className="w-full max-w-[200px] truncate overflow-hidden pl-1 text-ellipsis whitespace-nowrap">
      {value}
    </div>
  );

  return isOverflowing ? (
    <TooltipProvider>
      <Tooltip>
        <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
        <TooltipContent className="max-w-[600px] overflow-hidden p-0">
          <div className="max-h-[100px] overflow-x-auto overflow-y-auto p-2">
            <p className="break-all whitespace-pre-wrap">{value}</p>
          </div>
        </TooltipContent>
      </Tooltip>
    </TooltipProvider>
  ) : (
    TextContent
  );
}

const createColumn = <T extends SuricataRules['rules'][number]>(accessorKey: keyof T, title: string): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => <DataTableColumnHeader className="text-muted-foreground" column={column} title={title} />,
  cell: ({ row }) => {
    const value = String(row.getValue(accessorKey as string));
    return <TruncatedTextCell value={value} />;
  },
});

function StatusSwitchCell({
  status,
  sid,
  toggleMutation,
  userRole,
}: {
  status: 'Active' | 'Inactive';
  sid: string;
  toggleMutation: (params: { sids: string; toggleMode: 'toggle' }) => void;
  userRole: Role | undefined;
}) {
  const handleToggle = () => {
    toggleMutation({ sids: sid, toggleMode: 'toggle' });
  };

  return (
    <Switch
      disabled={!hasPermission(userRole, 'suricata_permissions', 'rules', 'put')}
      checked={status === 'Active'}
      onCheckedChange={handleToggle}
    />
  );
}

const createStatusButton = (
  toggleMutation: (params: { sids: string; toggleMode: 'toggle' }) => void,
  userRole: Role | undefined
): ColumnDef<SuricataRules['rules'][number]> => ({
  id: 'status',
  accessorKey: 'status',
  header: ({ column }) => <DataTableColumnHeader className="text-muted-foreground" column={column} title={'Status'} />,
  cell: ({ row }) => {
    const status = row.getValue('status') as 'Active' | 'Inactive';
    return (
      <StatusSwitchCell status={status} sid={row.original.SID} toggleMutation={toggleMutation} userRole={userRole} />
    );
  },
});

const actionsButton: ColumnDef<SuricataRules['rules'][number]> = {
  id: 'actions',
  cell: ({ row }) => {
    return <RulesDropdown sid={row.original.SID} />;
  },
};

const checkBox: ColumnDef<SuricataRules['rules'][number]> = {
  id: 'select',
  header: ({ table }) => (
    <Checkbox
      checked={table.getIsAllPageRowsSelected() || (table.getIsSomePageRowsSelected() && 'indeterminate')}
      onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
      aria-label="Select all"
    />
  ),
  cell: ({ row }) => (
    <Checkbox
      checked={row.getIsSelected()}
      onCheckedChange={(value) => row.toggleSelected(!!value)}
      aria-label="Select row"
    />
  ),
  enableSorting: false,
  enableHiding: false,
};

export const getSuricataRulesColumns = (
  toggleMutation: (params: { sids: string; toggleMode: 'toggle' }) => void,
  userRole: Role | undefined
): ColumnDef<SuricataRules['rules'][number]>[] => [
  checkBox,
  createColumn('SID', 'SID'),
  createColumn('action', 'Action'),
  createColumn('protocol', 'Protocol'),
  createColumn('source ip', 'Source IP'),
  createColumn('source port', 'Source Port'),
  createColumn('direction', 'Direction'),
  createColumn('destination ip', 'Destination IP'),
  createColumn('destination port', 'Destination Port'),
  createStatusButton(toggleMutation, userRole),
  actionsButton,
];
