import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { useTranslation } from 'react-i18next';
import { Ellipsis, Trash2, Pencil } from 'lucide-react';
import { useState } from 'react';
import { AlertDialog } from '@/components/ui/alert-dialog';
import { AddEditRulesetDialog } from '@/components/suricata/rules/rulesets/AddEditRulesetDialog';
import { useEditRuleset } from '@/components/suricata/rules/rulesets/queries';
import DeleteRulesetAlert from '@/components/suricata/rules/rulesets/DeleteRulesetAlert';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';
type RulesetOptionsProps = {
  id: number;
  name: string;
};

export function RulesetActionsDropdown({ id, name }: RulesetOptionsProps) {
  const { t } = useTranslation();
  const [isEditOpen, setIsEditOpen] = useState(false);
  const [isDeleteOpen, setIsDeleteOpen] = useState(false);
  const handleEdit = () => setIsEditOpen(true);
  const handleDelete = () => setIsDeleteOpen(true);
  const { mutate: editMutation } = useEditRuleset();
  const { user } = useAuth();

  const canEdit = hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'put');
  const canDelete = hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'delete');
  const canShowMenu = canEdit || canDelete;

  return (
    canShowMenu && (
      <>
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="ghost" size="sm" className="ml-auto hidden h-8 lg:flex">
              <Ellipsis />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            {canEdit && (
              <DropdownMenuItem className="cursor-pointer" onClick={handleEdit}>
                <Pencil />
                {t('Edit')}
              </DropdownMenuItem>
            )}
            {canDelete && (
              <DropdownMenuItem className="cursor-pointer" onClick={handleDelete}>
                <Trash2 />
                {t('Delete')}
              </DropdownMenuItem>
            )}
          </DropdownMenuContent>
        </DropdownMenu>
        <AddEditRulesetDialog
          isOpen={isEditOpen}
          onOpenChange={setIsEditOpen}
          rulesetId={id}
          onSubmit={(data, rulesetId) => {
            if (!rulesetId) return;
            editMutation({ ...data, id: rulesetId });
          }}
        />
        <AlertDialog open={isDeleteOpen} onOpenChange={setIsDeleteOpen}>
          <DeleteRulesetAlert id={id} name={name} />
        </AlertDialog>
      </>
    )
  );
}
