import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogTrigger,
  DialogClose,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useTranslation } from 'react-i18next';
import { Settings } from 'lucide-react';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { Form, FormControl, FormItem, FormLabel, FormMessage, FormDescription } from '@/components/ui/form';
import { useGetConfig, useUpdateRulesetInterval } from '@/components/suricata/config/queries';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';

export function RulesetUpdateIntervalDialog() {
  const { t } = useTranslation();
  const [open, setOpen] = useState(false);

  const { mutate: updateRulesetInterval } = useUpdateRulesetInterval();
  const { data: configData } = useGetConfig();

  const form = useForm({
    defaultValues: {
      timePeriod: configData?.ruleset_update_interval || 'daily',
    },
  });

  const handleOpenChange = (isOpen: boolean) => {
    setOpen(isOpen);
    if (isOpen) {
      form.reset({
        timePeriod: configData?.ruleset_update_interval || 'daily',
      });
    }
  };

  const onSubmit = form.handleSubmit((values) => {
    updateRulesetInterval(values.timePeriod);
    setOpen(false);
  });

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <Form {...form}>
        <form>
          <Tooltip>
            <TooltipTrigger asChild>
              <DialogTrigger asChild>
                <Button size="icon" variant="outline">
                  <Settings />
                </Button>
              </DialogTrigger>
            </TooltipTrigger>
            <TooltipContent>{t('Suricata Ruleset Update Interval Configuration')}</TooltipContent>
          </Tooltip>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>{t('Suricata Ruleset Update Interval Configuration')}</DialogTitle>
              <DialogDescription>{t('Configure your update interval for Suricata rulesets here')}</DialogDescription>
            </DialogHeader>
            <FormItem className="mb-5">
              <FormLabel>{t('Update Interval')}</FormLabel>
              <FormControl>
                <Select value={form.watch('timePeriod')} onValueChange={(value) => form.setValue('timePeriod', value)}>
                  <SelectTrigger className="w-full">
                    <SelectValue placeholder={t('Select an interval')} />
                  </SelectTrigger>
                  <SelectContent>
                    {[
                      { value: '5min', label: t('Every 5 Minutes') },
                      { value: 'hourly', label: t('Hourly') },
                      { value: 'daily', label: t('Daily') },
                      { value: 'weekly', label: t('Weekly') },
                      { value: 'monthly', label: t('Monthly') },
                    ].map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormDescription>
                {t('Select how often the Suricata rulesets should be updated automatically.')}
              </FormDescription>
              <FormMessage />
            </FormItem>
            <DialogFooter>
              <DialogClose asChild>
                <Button variant="secondary">{t('Close')}</Button>
              </DialogClose>
              <Button onClick={onSubmit}>{t('Save')}</Button>
            </DialogFooter>
          </DialogContent>
        </form>
      </Form>
    </Dialog>
  );
}
