import { DataTable } from '@/components/ui/DataTable/DataTable';
import { getRulesetColumns } from '@/components/suricata/rules/rulesets/rulesetColumns';
import { AddEditRulesetDialog } from '@/components/suricata/rules/rulesets/AddEditRulesetDialog';
import { useEffect, useState } from 'react';
import { useAddRuleset, useGetSuricataRulesets } from '@/components/suricata/rules/rulesets/queries';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import { Button } from '@/components/ui/button';
import { Info, Plus } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { SuricataRuleset } from './types';
import type { DataTableRowAction } from '@/components/ui/DataTable/DataTable';
import { useDeleteRuleset, usePatchRuleset } from '@/components/suricata/rules/rulesets/queries';
import { Switch } from '@/components/ui/switch';
import { Card, CardContent } from '@/components/ui/card';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import { useGetConfig, useUpdateAutoRuleset } from '../../config/queries';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';
import { RulesetUpdateIntervalDialog } from '@/components/suricata/rules/rulesets/RulesetUpdateIntervalDialog';

export function SuricataRulesets() {
  const { t } = useTranslation();
  const [isEditOpen, setEditOpen] = useState(false);
  const { mutate: addRulesetMutation } = useAddRuleset();
  const { data, isLoading } = useGetSuricataRulesets();
  const { mutate: deleteMutation } = useDeleteRuleset();
  const { mutate: patchRuleset } = usePatchRuleset();
  const { mutate: autoRuleset } = useUpdateAutoRuleset();
  const { data: configData } = useGetConfig();
  const { user } = useAuth();
  const [autoUpdate, setAutoUpdate] = useState<boolean>(false);
  const actions: DataTableRowAction<SuricataRuleset>[] = [];

  if (hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'delete')) {
    actions.push({
      label: t('Delete'),
      value: 'delete',
      onClick: (selectedRows) => {
        const ids = selectedRows.map((row) => row.id);
        deleteMutation(ids);
      },
    });
  }

  if (hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'put')) {
    actions.push(
      {
        label: t('Turn on'),
        value: 'on',
        onClick: (selectedRows) => {
          const ids = selectedRows.map((row) => row.id);
          patchRuleset({ ids, updates: { enabled: true } });
        },
      },
      {
        label: t('Turn off'),
        value: 'off',
        onClick: (selectedRows) => {
          const ids = selectedRows.map((row) => row.id);
          patchRuleset({ ids, updates: { enabled: false } });
        },
      }
    );
  }

  useEffect(() => {
    if (configData?.auto_update_rules !== undefined) {
      setAutoUpdate(Boolean(configData.auto_update_rules));
    }
  }, [configData]);

  if (isLoading || !data) return <DataTableSkeleton />;

  return (
    <div>
      {hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'post') && (
        <div className="flex items-center justify-between gap-4 pb-2">
          <Button
            onClick={() => {
              setEditOpen(true);
            }}
          >
            <Plus />
            {t('Add Ruleset')}
          </Button>
          <div className="flex items-center gap-2">
            <Card className="p-2">
              <CardContent className="flex items-center gap-x-2 px-1">
                <Switch
                  id="auto-update"
                  checked={autoUpdate}
                  onCheckedChange={(checked) => {
                    autoRuleset(checked);
                    setAutoUpdate(checked);
                  }}
                />
                <label className="text-muted-foreground text-sm font-bold">{t('Update Rulesets Automatically')}</label>
                <Tooltip>
                  <TooltipTrigger asChild>
                    <span className="cursor-pointer">
                      <Info className="h-5 w-5" />
                    </span>
                  </TooltipTrigger>
                  <TooltipContent side="top" align="center">
                    {t('Automatically updates your Suricata rules file using the selected rulesets.')}
                    <br />
                    {t('WARNING: Manual changes to suricata.rules will be overwritten.')}
                  </TooltipContent>
                </Tooltip>
              </CardContent>
            </Card>
            <RulesetUpdateIntervalDialog />
          </div>
        </div>
      )}
      <DataTable
        columns={getRulesetColumns(patchRuleset, user?.role, t)}
        data={data}
        usesRowSelection={true}
        selectedRowActions={actions}
      />
      <AddEditRulesetDialog
        isOpen={isEditOpen}
        onOpenChange={setEditOpen}
        onSubmit={(data) => {
          addRulesetMutation(data);
          setEditOpen(false);
        }}
      />
    </div>
  );
}
