import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { SuricataRuleset, SuricataRulesetFormValues } from '@/components/suricata/rules/rulesets/types';
import axios from '@/lib/axios';
import { toast } from 'sonner';
import { useTranslation } from 'react-i18next';

export function useGetSuricataRulesets() {
  return useQuery<SuricataRuleset[]>({
    queryKey: ['SuricataRulesets'],
    queryFn: async () => {
      const response = await axios.get<SuricataRuleset[]>('/api/v1/suricata/rulesets');
      return response.data;
    },
  });
}

export function useGetSuricataRuleset(id: number | null, enabled = true) {
  return useQuery<SuricataRuleset>({
    queryKey: ['SuricataRuleset', id],
    queryFn: async () => {
      const response = await axios.get<SuricataRuleset>(`/api/v1/suricata/rulesets/${id}`);
      return response.data;
    },
    enabled: enabled,
  });
}

type PatchRulesetParams = {
  ids: number[] | number;
  updates: Partial<SuricataRuleset>;
};

export function usePatchRuleset() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: async ({ ids, updates }: PatchRulesetParams) => {
      const idArray = Array.isArray(ids) ? ids : [ids];
      if (!updates || Object.keys(updates).length === 0) {
        throw new Error('No updates provided');
      }

      return Promise.all(idArray.map((id) => axios.patch(`/api/v1/suricata/rulesets/${id}`, updates)));
    },
    onMutate: async ({ ids, updates }) => {
      await queryClient.cancelQueries({ queryKey: ['SuricataRulesets'] });

      const previousData = queryClient.getQueryData<SuricataRuleset[]>(['SuricataRulesets']);

      queryClient.setQueryData(['SuricataRulesets'], (oldData: SuricataRuleset[] | undefined) => {
        if (!oldData) return oldData;

        const targetIds = Array.isArray(ids) ? ids : [ids];

        return oldData.map((ruleset) => {
          if (!targetIds.includes(ruleset.id)) return ruleset;

          return {
            ...ruleset,
            ...updates,
          };
        });
      });

      return { previousData };
    },
    onError: (_err, _vars, context) => {
      if (context?.previousData) {
        queryClient.setQueryData(['SuricataRulesets'], context.previousData);
      }
      toast.error(t('Failed to update rulesets.'));
    },
    onSuccess: () => {
      toast.success(t('Rulesets updated.'));
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['SuricataRulesets'] });
    },
  });
}

type useAddRulesetParams = SuricataRulesetFormValues;

type AddRulesetError = {
  response?: {
    status?: number;
    data?: {
      message?: string;
    };
  };
};

export function useAddRuleset() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (params: useAddRulesetParams) => {
      return axios.post('/api/v1/suricata/rulesets', params);
    },
    onError: (error: AddRulesetError) => {
      if (error?.response?.status === 422 && error.response.data?.message) {
        const message = error.response.data.message;
        const match = message.match(/Ruleset '(.+?)' already exists\./);
        if (match) {
          const rulesetName = match[1];
          toast.error(`${t('Failed to add ruleset. Ruleset')} '${rulesetName}' ${t('already exists.')}`);
          return;
        }
      }
      toast.error(t('Failed to add ruleset.'));
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SuricataRulesets'] });
      toast.success(t('Ruleset added.'));
    },
  });
}

export function useEditRuleset() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (params: SuricataRulesetFormValues & { id: number }) => {
      const { id, name: _name, ...restParams } = params;
      return axios.put(`/api/v1/suricata/rulesets/${id}`, restParams);
    },
    onError: () => {
      toast.error(t('Failed to update ruleset.'));
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SuricataRulesets'] });
      toast.success(t('Ruleset updated.'));
    },
  });
}

export function useDeleteRuleset() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: async (ids: number | number[]) => {
      const payload = Array.isArray(ids) ? { ids } : { ids: [ids] };

      return axios.delete(`/api/v1/suricata/rulesets`, {
        data: payload,
      });
    },
    onMutate: async (ids: number | number[]) => {
      await queryClient.cancelQueries({ queryKey: ['SuricataRulesets'] });

      const previousRulesets = queryClient.getQueryData<SuricataRuleset[]>(['SuricataRulesets']);

      const idArray = Array.isArray(ids) ? ids : [ids];

      if (previousRulesets) {
        queryClient.setQueryData(
          ['SuricataRulesets'],
          previousRulesets.filter((ruleset) => !idArray.includes(ruleset.id))
        );
      }

      return { previousRulesets };
    },
    onError: (_err, _ids, context) => {
      if (context?.previousRulesets) {
        queryClient.setQueryData(['SuricataRulesets'], context.previousRulesets);
      }
      toast.error(t('Failed to delete rulesets.'));
    },
    onSuccess: () => {
      toast.success(t('Rulesets deleted.'));
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['SuricataRulesets'] });
    },
  });
}
