import { ChevronDown, ChevronUp, InfoIcon } from "lucide-react";
import { Card, CardHeader, CardTitle, CardDescription, CardContent, CardFooter } from "./card";
import { Tooltip, TooltipTrigger, TooltipContent } from "./tooltip";
import { cn } from "@/lib/utils";
import { useState } from "react";
import { Button } from "./button";

interface InfoCardProps {
  header: string;
  data: string | number | React.ReactNode;
  unit?: string;
  description?: string;
  icon: React.ReactNode;
  tooltip: string;
  className?: string;
  cardContentClassName?: string;
  onClick?: () => void;
  expandable?: boolean;
  fullData?: string;
}

export default function InfoCard({ header, data, unit, description, icon, tooltip, className, cardContentClassName, onClick, expandable, fullData }: InfoCardProps) {
  const [isExpanded, setIsExpanded] = useState(false);

  const handleExpand = () => {
    setIsExpanded(!isExpanded);
  };

  return (
    <Card className={`${cn("flex-1 gap-2 h-full", className)}`} onClick={onClick}>
      <CardHeader>
        <CardTitle className="flex w-full items-center justify-between gap-2">
          <div className="flex items-center gap-2">
            <span className="text-primary">{icon}</span>
            <span className="font-medium text-lg">{header}</span>
          </div>
          <Tooltip>
            <TooltipTrigger asChild>
              <InfoIcon className="w-5 h-5" />
            </TooltipTrigger>
            <TooltipContent>{tooltip}</TooltipContent>
          </Tooltip>
        </CardTitle>
        <CardDescription>{description}</CardDescription>
      </CardHeader>
      <CardContent className={cn(`flex items-center gap-2 justify-center py-4 text-2xl font-bold ${expandable && (isExpanded ? 'h-fit' : 'h-64 overflow-hidden')}`, cardContentClassName)}>
        {typeof data === "string" && fullData && data !== fullData ? (
          <Tooltip>
            <TooltipTrigger asChild>
              <span className="truncate">{data}</span>
            </TooltipTrigger>
            <TooltipContent>{fullData}</TooltipContent>
          </Tooltip>
        ) : (
          <>
            {data}
            {unit && <span className="text-sm text-muted-foreground">{unit}</span>}
          </>
        )}
      </CardContent>
      <CardFooter>
        {expandable && (
          <Button variant="ghost" size="icon" onClick={handleExpand} className="ml-auto">
            {isExpanded ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
          </Button>
        )}
      </CardFooter>
    </Card>
  );
};