import React, { useState } from "react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";
import { Clock } from "lucide-react";
import {
    Popover,
    PopoverContent,
    PopoverTrigger,
} from "@/components/ui/popover";
import { ScrollArea } from "@/components/ui/scroll-area";

interface SingleTimePickerProps {
    value: string;
    onChange: (value: string) => void;
    className?: string;
}

interface RangeTimePickerProps {
    value: string[];
    onChange: (value: string[]) => void;
    className?: string;
    mode: 'range' | 'default';
}

type TimePickerProps = SingleTimePickerProps | RangeTimePickerProps;

const TimePickerSelect: React.FC<{
    value: string;
    onChange: (time: string) => void;
}> = ({ value, onChange }) => {
    const [period, setPeriod] = useState(
        value ? (parseInt(value.split(":")[0]) >= 12 ? "PM" : "AM") : "AM"
    );
    const hours = Array.from({ length: 12 }, (_, i) =>
        (i + 1).toString().padStart(2, "0")
    );
    const minutes = Array.from({ length: 60 }, (_, i) =>
        i.toString().padStart(2, "0")
    );

    const handleHourClick = (hour: string) => {
        const hourNum = parseInt(hour);
        const adjustedHour =
            period === "PM"
                ? hourNum === 12
                    ? 12
                    : hourNum + 12
                : hourNum === 12
                ? 0
                : hourNum;
        const currentMinute = value?.split(":")[1] || "00";
        onChange(
            `${adjustedHour.toString().padStart(2, "0")}:${currentMinute}`
        );
    };

    const handlePeriodClick = (newPeriod: string) => {
        setPeriod(newPeriod);
        if (!value) return;

        const [hour, minute] = value.split(":");
        const hourNum = parseInt(hour);

        if (newPeriod === "AM" && hourNum >= 12) {
            onChange(`${(hourNum - 12).toString().padStart(2, "0")}:${minute}`);
        } else if (newPeriod === "PM" && hourNum < 12) {
            onChange(`${(hourNum + 12).toString().padStart(2, "0")}:${minute}`);
        }
    };

    return (
        <div className="flex gap-2 p-2">
            <ScrollArea className="h-[200px] w-[60px]">
                <div className="flex flex-col pl-2 py-1 w-11">
                    {hours.map((hour) => (
                        <Button
                            key={hour}
                            variant="ghost"
                            className={cn(
                                "justify-start rounded-sm px-2",
                                value &&
                                    parseInt(value.split(":")[0]) % 12 ===
                                        parseInt(hour) % 12 &&
                                    "bg-accent text-accent-foreground"
                            )}
                            onClick={(e) => {
                                e.preventDefault();
                                handleHourClick(hour);
                            }}
                            type="button"
                        >
                            {hour}
                        </Button>
                    ))}
                </div>
            </ScrollArea>
            <ScrollArea className="h-[200px] w-[60px]">
                <div className="flex flex-col pl-2 py-1 w-11">
                    {minutes.map((minute) => (
                        <Button
                            key={minute}
                            variant="ghost"
                            className={cn(
                                "justify-start rounded-sm px-2",
                                value?.endsWith(minute) &&
                                    "bg-accent text-accent-foreground"
                            )}
                            onClick={(e) => {
                                e.preventDefault();
                                const currentHour =
                                    value?.split(":")[0] || "00";
                                onChange(`${currentHour}:${minute}`);
                            }}
                            type="button"
                        >
                            {minute}
                        </Button>
                    ))}
                </div>
            </ScrollArea>
            <div className="flex flex-col gap-2">
                {["AM", "PM"].map((p) => (
                    <Button
                        key={p}
                        variant="ghost"
                        className={cn(
                            "justify-start rounded-sm px-2",
                            period === p && "bg-accent text-accent-foreground"
                        )}
                        onClick={(e) => {
                            e.preventDefault();
                            handlePeriodClick(p);
                        }}
                        type="button"
                    >
                        {p}
                    </Button>
                ))}
            </div>
        </div>
    );
};

const TimePicker = (props: TimePickerProps) => {
    const [openPopover, setOpenPopover] = useState<number | null>(null);

    if ('mode' in props && props.mode === 'range') {
        const { value, onChange, className } = props;
        
        return (
            <div className={cn("flex items-center gap-2", className)}>
                <Popover
                    open={openPopover === 0}
                    onOpenChange={(open) => setOpenPopover(open ? 0 : null)}
                >
                    <div className="relative">
                        <Input
                            type="time"
                            value={value[0] || ""}
                            onChange={(e) => onChange([e.target.value, value[1]])}
                            className="w-[130px] pr-8 [&::-webkit-calendar-picker-indicator]:hidden"
                        />
                        <PopoverTrigger asChild>
                            <Button
                                variant="ghost"
                                size="icon"
                                className="absolute right-0 top-0 size-7 m-1 hover:bg-transparent"
                                type="button"
                            >
                                <Clock className="h-4 w-4 text-muted-foreground" />
                            </Button>
                        </PopoverTrigger>
                    </div>
                    <PopoverContent className="w-[180px] p-0">
                        <TimePickerSelect
                            value={value[0] || ""}
                            onChange={(time) => onChange([time, value[1]])}
                        />
                    </PopoverContent>
                </Popover>
                <span className="text-muted-foreground px-2">to</span>
                <Popover
                    open={openPopover === 1}
                    onOpenChange={(open) => setOpenPopover(open ? 1 : null)}
                >
                    <div className="relative">
                        <Input
                            type="time"
                            value={value[1] || ""}
                            onChange={(e) => onChange([value[0], e.target.value])}
                            className="w-[130px] pr-8 [&::-webkit-calendar-picker-indicator]:hidden"
                        />
                        <PopoverTrigger asChild>
                            <Button
                                variant="ghost"
                                size="icon"
                                className="absolute right-0 top-0 size-7 m-1 hover:bg-transparent"
                                type="button"
                            >
                                <Clock className="h-4 w-4 text-muted-foreground" />
                            </Button>
                        </PopoverTrigger>
                    </div>
                    <PopoverContent className="w-[180px] p-0">
                        <TimePickerSelect
                            value={value[1] || ""}
                            onChange={(time) => onChange([value[0], time])}
                        />
                    </PopoverContent>
                </Popover>
            </div>
        );
    }

    // Single time picker
    const { value, onChange, className } = props;
    
    return (
        <div className={cn("relative", className)}>
            <Input
                type="time"
                value={value}
                onChange={(e) => {
                    if (typeof onChange === "function" && typeof value === "string") {
                        (onChange as (value: string) => void)(e.target.value);
                    }
                }}
                className="w-[130px] pr-8 [&::-webkit-calendar-picker-indicator]:hidden"
            />
            <Popover
                open={openPopover === 0}
                onOpenChange={(open) => setOpenPopover(open ? 0 : null)}
            >
                <PopoverTrigger asChild>
                    <Button
                        variant="ghost"
                        size="icon"
                        className="absolute right-0 top-0 size-7 m-1 hover:bg-transparent"
                        type="button"
                    >
                        <Clock className="h-4 w-4 text-muted-foreground" />
                    </Button>
                </PopoverTrigger>
                <PopoverContent className="w-[180px] p-0">
                    <TimePickerSelect
                        value={typeof value === "string" ? value : ""}
                        onChange={(time) => {
                            if (typeof onChange === "function") {
                                (onChange as (value: string) => void)(time);
                            }
                        }}
                    />
                </PopoverContent>
            </Popover>
        </div>
    );
};

export default TimePicker;