import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { UserFormData, BaseUser } from '@/components/user/types';
import { useNavigate } from 'react-router-dom';

export function useGetUsers() {
  return useQuery<BaseUser[], AxiosError>({
    queryKey: ['users'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/users');
      return response.data;
    },
    retry: false,
  });
}

export function useCreateUser() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const navigate = useNavigate();

  return useMutation({
    mutationFn: (data: UserFormData) => {
      return axios.post('/api/v1/users', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      navigate('/admin/users');
      toast(t('User Created.'));
    },
    onError: () => {
      toast(t('Failed to create user.'), {
        description: t('Please try again.'),
      });
    },
  });
}

export function useEditUser() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ data, id }: { data: UserFormData; id: number }) => {
      return axios.put(`/api/v1/users/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      toast(t('User edited.'));
    },
    onError: () => {
      toast(t('Failed to edit user.'), {
        description: t('Please try again.'),
      });
    },
  });
}

export function useDeleteUser() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => axios.delete(`/api/v1/users/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      toast(t('User deleted.'));
    },
    onError: (error: { response?: { status?: number } }) => {
      if (error?.response?.status === 403) {
        toast(t('You cannot delete your own account.'));
      } else {
        toast(t('Failed to delete user.'), {
          description: t('Please try again'),
        });
      }
    },
  });
}
