import { useState, Fragment } from 'react';
import { useTranslation } from 'react-i18next';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { BracesIcon, ChevronDown, ChevronRight, ClipboardListIcon, ListCollapseIcon } from 'lucide-react';
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetDescription } from '@/components/ui/sheet';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { ScrollArea, ScrollBar } from '@/components/ui/scroll-area';
import { NestedData } from '@/components/wireshark/types';
import { useGetPacketDetails } from '@/components/wireshark/queries';
import PacketSummary from '@/components/wireshark/PacketSummary';

interface NestedTableProps {
  data: NestedData;
  level?: number;
}

export const NestedTable: React.FC<NestedTableProps> = ({ data, level = 0 }) => {
  const [openKeys, setOpenKeys] = useState<Record<string, boolean>>({});

  if (typeof data !== 'object' || data === null) {
    return null;
  }

  const toggleOpen = (key: string) => {
    setOpenKeys((prev) => ({ ...prev, [key]: !prev[key] }));
  };

  return (
    <>
      {Object.entries(data).map(([key, value]) => {
        const isObject = typeof value === 'object' && value !== null;
        const uniqueKey = `${level}-${key}`;
        const indentStyle = { paddingLeft: `${level * 16}px` };

        if (isObject) {
          return (
            <Fragment key={uniqueKey}>
              <TableRow
                className="bg-muted/30 hover:bg-muted/40 cursor-pointer transition-colors"
                onClick={() => toggleOpen(uniqueKey)}
              >
                <TableCell style={indentStyle} className="text-primary w-1/2 text-sm font-semibold">
                  {openKeys[uniqueKey] ? (
                    <ChevronDown className="mr-2 inline" />
                  ) : (
                    <ChevronRight className="mr-2 inline" />
                  )}
                  {key}
                </TableCell>
                <TableCell className="text-muted-foreground w-1/2 text-sm italic">
                  {Array.isArray(value) ? `[${value.length} items]` : '{...}'}
                </TableCell>
              </TableRow>
              {openKeys[uniqueKey] &&
                (Array.isArray(value) || (value && typeof value === 'object' && !Array.isArray(value)) ? (
                  <NestedTable data={value as NestedData} level={level + 1} />
                ) : null)}
            </Fragment>
          );
        }

        return (
          <TableRow key={uniqueKey}>
            <TableCell style={indentStyle} className="w-1/2 text-sm font-normal">
              {key}
            </TableCell>
            <TableCell className="text-foreground w-1/2 max-w-[300px] p-0 text-sm">
              {typeof value === 'string' && value.length > 80 ? (
                <ScrollArea className="w-full p-2">
                  <span className="break-all">{value}</span>
                  <ScrollBar orientation="horizontal" className="mt-2" />
                </ScrollArea>
              ) : (
                <span className="block p-2">{String(value)}</span>
              )}
            </TableCell>
          </TableRow>
        );
      })}
    </>
  );
};

interface PacketDetailSheetProps {
  isOpen: boolean;
  onOpenChange: (open: boolean) => void;
  framenumber: number;
  filename?: string;
}

export function PacketDetailSheet({ framenumber, isOpen, onOpenChange, filename }: PacketDetailSheetProps) {
  const { t } = useTranslation();

  const { data, isLoading, error } = useGetPacketDetails(framenumber, filename);

  type PacketDetail = Record<string, unknown>;
  let parsedPacketDetails: PacketDetail[] = [];
  let packetDetails: unknown = [];

  if (!isLoading && !error && data) {
    packetDetails = data.packetsDetails;
    if (Array.isArray(packetDetails)) {
      parsedPacketDetails = packetDetails;
    } else if (typeof packetDetails === 'string') {
      try {
        parsedPacketDetails = JSON.parse(packetDetails);
      } catch (e) {
        console.error('Failed to parse packetDetails JSON', e);
        parsedPacketDetails = [];
      }
    } else {
      parsedPacketDetails = [];
    }
  }

  return (
    <Sheet open={isOpen} onOpenChange={onOpenChange}>
      <SheetContent className="p-0 sm:max-w-[800px]">
        <div className="h-full p-4">
          <SheetHeader>
            <SheetTitle>Frame {framenumber}</SheetTitle>
            <SheetDescription>Verbose details of the packet captured in Wireshark.</SheetDescription>
          </SheetHeader>
          <div className="mt-4 h-full">
            <Tabs defaultValue="summary" className="h-full">
              <div className="mb-2 flex items-center space-x-2">
                <TabsList className="w-full">
                  <TabsTrigger value="summary" className="w-1/2 gap-1.5">
                    <ClipboardListIcon className="h-4 w-4" />
                    {t('Summary')}
                  </TabsTrigger>
                  <TabsTrigger value="details" className="w-1/2 gap-1.5">
                    <ListCollapseIcon className="h-4 w-4" />
                    {t('Advanced Details')}
                  </TabsTrigger>
                  <TabsTrigger value="json" className="w-1/2 gap-1.5">
                    <BracesIcon className="h-4 w-4" />
                    {t('Raw JSON')}
                  </TabsTrigger>
                </TabsList>
              </div>
              <ScrollArea className="h-[calc(100%-150px)]">
                <TabsContent value="summary">
                  <PacketSummary packetSummary={data?.packetsSummary} />
                </TabsContent>

                <TabsContent value="details">
                  {isLoading ? (
                    <div className="p-4 text-center">{t('Loading...')}</div>
                  ) : error ? (
                    <div className="p-4 text-center text-red-500">{t('Error loading packet details')}</div>
                  ) : (
                    <Table className="w-full">
                      <TableHeader>
                        <TableRow>
                          <TableHead>{t('Field')}</TableHead>
                          <TableHead>{t('Value')}</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {parsedPacketDetails.map((item, index) => (
                          <NestedTable key={index} data={item} />
                        ))}
                      </TableBody>
                    </Table>
                  )}
                </TabsContent>

                <TabsContent value="json">
                  {isLoading ? (
                    <div className="p-4 text-center">{t('Loading...')}</div>
                  ) : error ? (
                    <div className="p-4 text-center text-red-500">{t('Error loading packet details')}</div>
                  ) : (
                    <pre className="bg-muted rounded-md p-4 font-mono text-sm break-all whitespace-pre-wrap">
                      {JSON.stringify(parsedPacketDetails, null, 2)}
                    </pre>
                  )}
                </TabsContent>
              </ScrollArea>
            </Tabs>
          </div>
        </div>
      </SheetContent>
    </Sheet>
  );
}
