import { DataTable } from '@/components/ui/DataTable/DataTable';
import { columns } from '@/components/wireshark/capturehistory/columns';
import { useGetWiresharkCaptures } from '@/components/wireshark/capturehistory/queries';
import React from 'react';
import DataTableSkeleton from '@/components/ui/DataTable/DataTableSkeleton';
import DeleteCaptureDialog from '@/components/wireshark/capturehistory/DeleteCaptureDialog';
import ImportCaptureDialog from '@/components/wireshark/capturehistory/ImportCaptureDialog';
import { useTranslation } from 'react-i18next';
import { hasPermission } from '@/components/role/utils';
import { useAuth } from '@/contexts/AuthContext';
import { useSearchParams } from 'react-router';
import { WiresharkCapture } from '@/components/wireshark/capturehistory/types';

export default function CaptureHistoryDataTable() {
  const { data = [], isLoading, isError, error } = useGetWiresharkCaptures();
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [idsToDelete, setIdsToDelete] = React.useState<number[]>([]);
  const { t } = useTranslation();
  const { user } = useAuth();
  const [searchParams] = useSearchParams();
  const filter = searchParams.get('filter') ?? '';

  const actions = [];

  if (hasPermission(user?.role, 'wireshark_permissions', 'pcaps', 'delete')) {
    actions.push({
      label: t('Delete'),
      value: 'delete',
      onClick: async (selectedRows: WiresharkCapture[]) => {
        setIdsToDelete(selectedRows.map((row) => row.id));
        setDeleteDialogOpen(true);
      },
    });
  }

  if (isError) {
    return (
      <div>
        {t('Failed to load captures')}: {error.message}
      </div>
    );
  }

  return (
    <div>
      <div className="mb-4 flex items-center justify-between">
        <h2 className="text-xl font-semibold">{t('Capture History')}</h2>
        {hasPermission(user?.role, 'wireshark_permissions', 'pcaps', 'post') && <ImportCaptureDialog />}
      </div>
      {!isLoading && (
        <DataTable
          columns={columns}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultGlobalFilter={filter}
          data={data}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <DeleteCaptureDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteCaptureDialog>
    </div>
  );
}
