import { ColumnDef, Row } from '@tanstack/react-table';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { Checkbox } from '@/components/ui/checkbox';
import { Button } from '@/components/ui/button';
import DeleteCaptureDialog from '@/components/wireshark/capturehistory/DeleteCaptureDialog';
import { useTranslation } from 'react-i18next';
import { useDownloadCapture, useScanWithSuricata } from '@/components/wireshark/capturehistory/queries';
import { useNavigate } from 'react-router-dom';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Trash2, Eye, Download, MoreHorizontal, GlobeLockIcon } from 'lucide-react';
import { WiresharkCapture } from '@/components/wireshark/capturehistory/types';
import { useState } from 'react';
import { toast } from 'sonner';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { useAuth } from '@/contexts/AuthContext';
import { hasPermission } from '@/components/role/utils';

function ViewCell({ row }: { row: Row<WiresharkCapture> }) {
  const capture = row.original;
  const navigate = useNavigate();
  const { t } = useTranslation();

  const handleViewClick = () => {
    navigate(`/wireshark/${capture.id}`);
  };

  return (
    <TooltipProvider>
      <Tooltip>
        <TooltipTrigger asChild>
          <Button variant="ghost" className="h-8 w-8 p-0" onClick={handleViewClick}>
            <Eye />
            <span className="sr-only">{t('View Captured Packets')}</span>
          </Button>
        </TooltipTrigger>
        <TooltipContent side="right">
          <p>{t('View Captured Packets')}</p>
        </TooltipContent>
      </Tooltip>
    </TooltipProvider>
  );
}

function ActionCell({ row }: { row: Row<WiresharkCapture> }) {
  const capture = row.original;
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const { mutate: downloadCapture } = useDownloadCapture();
  const { mutate: scanWithSuricata } = useScanWithSuricata();
  const { t } = useTranslation();
  const { user } = useAuth();

  const handleDownload = () => {
    downloadCapture(capture.id, {
      onSuccess: ({ blob, captureId }) => {
        const url = window.URL.createObjectURL(new Blob([blob]));
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `capture_${captureId}.pcap`);
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(url);
        toast.success(t('Capture file downloaded successfully.'));
      },
      onError: () => {
        toast.error(t('Failed to download capture file.'));
      },
    });
  };

  const handleSuricataScan = () => {
    scanWithSuricata(capture.id);
  };

  return (
    <>
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost" className="h-8 w-8 p-0">
            <span className="sr-only">Open menu</span>
            <MoreHorizontal className="h-4 w-4" />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="center">
          <DropdownMenuItem onClick={handleDownload}>
            <Download className="mr-2 h-4 w-4" />
            {t('Download PCAP')}
          </DropdownMenuItem>
          {hasPermission(user?.role, 'suricata_permissions', 'scan_pcap') && (
            <DropdownMenuItem onClick={handleSuricataScan}>
              <GlobeLockIcon className="mr-2 h-4 w-4" />
              {t('Scan With Suricata')}
            </DropdownMenuItem>
          )}
          {hasPermission(user?.role, 'wireshark_permissions', 'pcaps', 'delete') && (
            <>
              <DropdownMenuSeparator />
              <DropdownMenuItem onClick={() => setDeleteDialogOpen(true)}>
                <Trash2 className="mr-2 h-4 w-4 text-red-500" />
                <span className="text-red-500">{t('Delete')}</span>
              </DropdownMenuItem>
            </>
          )}
        </DropdownMenuContent>
      </DropdownMenu>

      <DeleteCaptureDialog ids={[capture.id]} open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <span />
      </DeleteCaptureDialog>
    </>
  );
}

export const columns: ColumnDef<WiresharkCapture>[] = [
  {
    id: 'select',
    header: ({ table }) => (
      <Checkbox
        checked={table.getIsAllPageRowsSelected() || (table.getIsSomePageRowsSelected() && 'indeterminate')}
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
    ),
    cell: ({ row }) => (
      <Checkbox
        checked={row.getIsSelected()}
        onCheckedChange={(value) => row.toggleSelected(!!value)}
        aria-label="Select row"
      />
    ),
    enableSorting: false,
    enableHiding: false,
  },
  {
    id: 'ID',
    accessorKey: 'id',
    header: ({ column }) => <DataTableColumnHeader column={column} title="ID" />,
  },
  {
    id: 'view',
    cell: ({ row }) => <ViewCell row={row} />,
  },
  {
    id: 'file name',
    accessorKey: 'file_name',
    header: ({ column }) => <DataTableColumnHeader column={column} title="File Name" />,
  },
  {
    id: 'interface',
    accessorKey: 'interface',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Interface" />,
  },
  {
    id: 'started at',
    accessorKey: 'started_at',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Started At" />,
    cell: ({ row }) => {
      const value = row.original.started_at;
      if (value !== null) {
        return new Date(value).toLocaleString();
      } else {
        return 'In Progress';
      }
    },
  },
  {
    id: 'finished at',
    accessorKey: 'finished_at',
    header: ({ column }) => <DataTableColumnHeader column={column} title="Finished At" />,
    cell: ({ row }) => {
      const value = row.original.finished_at;
      if (value !== null) {
        return new Date(value).toLocaleString();
      } else {
        return 'In Progress';
      }
    },
  },
  {
    id: 'actions',
    enableSorting: false,
    cell: ({ row }) => <ActionCell row={row} />,
  },
];
