import { ColumnDef } from '@tanstack/react-table';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import { Packet } from '@/components/wireshark/captures/types';
import { useState, useEffect, useRef } from 'react';
import TextWithCopyButton from '@/components/ui/TextWithCopyButton';
import { TFunction } from 'i18next';

function InfoCell({ value }: { value: string | undefined }) {
  const spanRef = useRef<HTMLSpanElement>(null);
  const [isTruncated, setIsTruncated] = useState(false);

  useEffect(() => {
    const el = spanRef.current;
    if (el) {
      setIsTruncated(el.scrollWidth > el.clientWidth);
    }
  }, [value]);

  if (!value) return null;

  const content = (
    <span ref={spanRef} className="inline-block max-w-[200px] cursor-pointer truncate">
      {value}
    </span>
  );

  return isTruncated ? (
    <Tooltip>
      <TooltipTrigger asChild>{content}</TooltipTrigger>
      <TooltipContent className="max-w-[400px] p-2 text-center text-sm" side="top" align="center">
        <span className="block break-words whitespace-pre-wrap">{value}</span>
      </TooltipContent>
    </Tooltip>
  ) : (
    content
  );
}

export const columns = (t: TFunction): ColumnDef<Packet>[] => [
  {
    id: 'Frame Number',
    accessorKey: 'frameNumber',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="NO." />;
    },
  },
  {
    id: 'time',
    accessorKey: 'time',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Time" />;
    },
  },
  {
    id: 'source',
    accessorKey: 'source',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Source" />;
    },
    cell: ({ row }) => {
      return row.original.source ? <TextWithCopyButton text={row.original.source} /> : t('Unknown');
    },
  },
  {
    id: 'destination',
    accessorKey: 'destination',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Destination" />;
    },
    cell: ({ row }) => {
      return row.original.destination ? <TextWithCopyButton text={row.original.destination} /> : t('Unknown');
    },
  },
  {
    id: 'protocol',
    accessorKey: 'protocol',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Protocol" />;
    },
  },
  {
    id: 'length',
    accessorKey: 'length',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Length" />;
    },
  },
  {
    id: 'info',
    accessorKey: 'info',
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Info" />;
    },
    cell: ({ getValue }) => {
      const value = getValue<string>();
      return <InfoCell value={value} />;
    },
  },
];
