import { useQuery } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { Packet } from '@/components/wireshark/captures/types';

interface PacketsResponse {
  file_name: string;
  packets: Packet[];
  total: number;
}

interface UseHistoryParams {
  id: string;
  pageIndex: number;
  pageSize: number;
  sortBy: string;
  sortOrder: 'asc' | 'desc';
  filter?: string;
}

export function useGetPackets({ id, pageIndex, pageSize, sortBy, sortOrder, filter }: UseHistoryParams) {
  return useQuery<PacketsResponse, AxiosError>({
    queryKey: ['wiresharkCapture', id, pageIndex, pageSize, sortBy, sortOrder, filter],
    queryFn: async () => {
      const response = await axios.get<PacketsResponse>(`/api/v1/wireshark/captures/${id}/packets`, {
        params: {
          page: pageIndex + 1,
          per_page: pageSize,
          sortBy,
          sortOrder,
          ...(filter ? { filter } : {}),
        },
        paramsSerializer: (params) =>
          Object.entries(params)
            .map(([key, value]) => `${key}=${encodeURIComponent(value as string)}`)
            .join('&'),
      });
      return response.data;
    },
    enabled: !!id,
  });
}
