import { Bar, BarChart, CartesianGrid, XAxis, Cell, YAxis } from 'recharts';
import React from 'react';
import { useTranslation } from 'react-i18next';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { ChartConfig, ChartContainer, ChartTooltip, ChartTooltipContent } from '@/components/ui/chart';
import { PacketSizeBucket } from '@/components/wireshark/capturesummary/types';
import { useSearchParams } from 'react-router';

interface PacketSizeBarChartProps {
  packetSizeDistribution: PacketSizeBucket[];
}

const baseChartConfig = {
  packet_count: {
    label: 'Packet Count',
  },
} satisfies ChartConfig;

export function PacketSizeBarChart({ packetSizeDistribution }: PacketSizeBarChartProps) {
  const { t } = useTranslation();
  const [_, setSearchParams] = useSearchParams();

  const packetSizes = React.useMemo(() => {
    return packetSizeDistribution.map((item, index) => ({
      range: item.range,
      packet_count: item.count,
      fill: `var(--chart-${(index % 10) + 1})`,
    }));
  }, [packetSizeDistribution]);

  const handleBarClick = (data: PacketSizeBucket) => {
    let filter = '';
    if (data.range.includes('+')) {
      const lowerBound = data.range.replace('+', '').trim();
      filter = `frame.len >= ${lowerBound}`;
    } else {
      const range = data.range.split('-');
      const lowerBound = range[0];
      const upperBound = range[1];
      filter = `frame.len >= ${lowerBound} && frame.len <= ${upperBound}`;
    }
    setSearchParams({ tab: 'capture', filter: filter });
  };

  const dynamicChartConfig: ChartConfig = React.useMemo(() => {
    return {
      ...baseChartConfig,
      ...packetSizes.reduce((acc, item) => {
        acc[item.range] = {
          label: item.range,
          color: item.fill,
        };
        return acc;
      }, {} as ChartConfig),
    };
  }, [packetSizes]);

  return (
    <Card className="flex flex-col">
      <CardHeader className="items-center pb-0">
        <CardTitle>{t('Packet Size Distribution')}</CardTitle>
        <CardDescription>{t('Distribution of packets by size range')}</CardDescription>
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        <ChartContainer config={dynamicChartConfig} className="mx-auto aspect-auto h-[300px] w-full">
          <BarChart data={packetSizes} margin={{ top: 10, right: 10, left: 10, bottom: 10 }}>
            <CartesianGrid vertical={false} strokeDasharray="3 3" />
            <XAxis
              dataKey="range"
              tickLine={false}
              axisLine={false}
              tickMargin={8}
              angle={-45}
              textAnchor="end"
              height={75}
              label={{
                value: 'Bytes',
                position: 'insideBottom',
                offset: -5,
                style: { textAnchor: 'middle' },
              }}
            />
            <YAxis
              label={{
                value: 'Packets',
                angle: -90,
                position: 'insideLeft',
                offset: 0,
                style: { textAnchor: 'middle' },
              }}
              tickLine={false}
              axisLine={false}
            />
            <ChartTooltip content={<ChartTooltipContent className="w-[150px]" nameKey="packet_count" />} />
            <Bar className="cursor-pointer" dataKey="packet_count" onClick={handleBarClick}>
              {packetSizes.map((entry, index) => (
                <Cell key={`cell-${index}`} fill={entry.fill} />
              ))}
            </Bar>
          </BarChart>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col gap-2 text-sm" />
    </Card>
  );
}
