import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { useEcho } from '@laravel/echo-react';
import { toast } from 'sonner';
import { useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { useNavigate } from 'react-router-dom';
import { Card } from '@/components/ui/card';
import { Server, Hash, FileText, Clock, LoaderCircle, Eye, CheckCircle, XCircle, Download } from 'lucide-react';
import { Badge } from '@/components/ui/badge';
import type { VariantProps } from 'class-variance-authority';
import { badgeVariants } from '@/components/ui/badge';
import { useTranslation } from 'react-i18next';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tooltip, TooltipTrigger, TooltipContent } from '@/components/ui/tooltip';
import {
  useGetNetworkInterfaces,
  useStartCapture,
  useStopCapture,
  useGetLatestCapture,
} from '@/components/wireshark/livecapture/queries';
import { useDownloadCapture } from '@/components/wireshark/capturehistory/queries';
import { NetworkInterface } from '@/components/wireshark/livecapture/types';
import InfoCard from '@/components/ui/InfoCard';
import { Skeleton } from '@/components/ui/skeleton';
import { Duration } from '@/components/ui/Duration';

const FormSchema = z.object({
  networkInterface: z.string().min(1, 'Please select a network interface to capture.'),
  duration: z.number().min(1, 'Please select a capture duration.'),
});

type FormValues = z.infer<typeof FormSchema>;
type BadgeVariant = VariantProps<typeof badgeVariants>['variant'];

export function WiresharkCapture() {
  const queryClient = useQueryClient();
  const navigate = useNavigate();
  const { t } = useTranslation();

  const form = useForm<FormValues>({
    resolver: zodResolver(FormSchema),
    defaultValues: {
      networkInterface: '',
      duration: undefined,
    },
  });

  const { data: interfacesData } = useGetNetworkInterfaces();
  const interfaces: NetworkInterface[] = interfacesData || [];

  const startCaptureMutation = useStartCapture();
  const stopCaptureMutation = useStopCapture();
  const { mutate: downloadCapture } = useDownloadCapture();
  const { data: activeCapture, isLoading } = useGetLatestCapture();

  // Live updates from Echo broadcast
  useEcho(`NNA.Users`, 'PacketCaptured', () => {
    queryClient.invalidateQueries({ queryKey: ['activecapture'] });
  });

  const capture = activeCapture ?? {
    id: null,
    interface: '',
    status: '',
    file_name: '',
    started_at: '',
    finished_at: null,
    pid: null,
  };

  const fileNameOnly = capture.file_name ? capture.file_name.split(/[/\\]/).pop() : ' ';

  const handleStartCapture = (values: FormValues) => {
    if (!values.networkInterface) {
      toast.warning(t('Please select a network interface before starting the capture.'));
      return;
    }
    if (!values.duration) {
      toast.warning(t('Please select a duration before starting the capture.'));
      return;
    }
    if (capture.status === 'in_progress') {
      toast.warning(t('Capture already in progress. Stop or restart before starting a new one.'));
      return;
    }
    startCaptureMutation.mutate(
      { networkInterface: values.networkInterface, duration: values.duration },
      {
        onSuccess: () => {
          queryClient.invalidateQueries({ queryKey: ['activecapture'] });
        },
      }
    );
  };

  const handleStopCapture = () => {
    if (!capture.id) {
      toast.error(t('No active capture to stop.'));
      return;
    }

    stopCaptureMutation.mutate(capture.id);
  };

  const handleView = () => {
    navigate(`/wireshark/${capture.id}`);
  };

  const handleDownload = (captureId: number) => {
    if (!captureId) {
      toast.error(t('No capture ID available for download.'));
      return;
    }
    downloadCapture(captureId, {
      onSuccess: ({ blob, captureId }) => {
        const url = window.URL.createObjectURL(new Blob([blob]));
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `capture_${captureId}.pcap`);
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(url);
        toast.success(t('Capture file downloaded successfully.'));
      },
      onError: () => {
        toast.error(t('Failed to download capture file.'));
      },
    });
  };

  let variant: BadgeVariant = 'default';
  let status = 'Running';

  if (capture.status === 'completed') {
    variant = 'green';
    status = 'Completed';
  } else if (capture.status === 'in_progress') {
    variant = 'blue';
    status = 'Running';
  } else if (capture.status === 'pending') {
    variant = 'blue';
    status = 'Pending';
  } else if (capture.status === 'failed') {
    variant = 'red';
    status = 'Failed';
  } else {
    status = 'No Capture Running';
  }

  const durationOptions = [
    { value: 10, label: t('10 Seconds') },
    { value: 30, label: t('30 Seconds') },
    { value: 60, label: t('1 Minute') },
    { value: 180, label: t('3 Minutes') },
    { value: 300, label: t('5 Minutes') },
  ];

  const formatDateTime = (dateTime: string) => {
    return new Date(dateTime.replace(' ', 'T')).toLocaleString();
  };

  return (
    <div className="relative hidden h-full flex-1 flex-col space-y-8 py-2 md:flex">
      <Form {...form}>
        <form onSubmit={form.handleSubmit(handleStartCapture)} className="w-full max-w-xl space-y-4">
          <div className="flex flex-wrap items-end gap-3">
            <FormField
              control={form.control}
              name="networkInterface"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Network Interface')}</FormLabel>
                  <Select onValueChange={field.onChange} value={field.value}>
                    <FormControl>
                      <SelectTrigger className="w-50">
                        <SelectValue placeholder={t('Select an interface')} />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {interfaces.map((iface) => (
                        <SelectItem key={iface.id} value={iface.name}>
                          {iface.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="duration"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t('Duration')}</FormLabel>
                  <Select
                    value={field.value ? String(field.value) : ''}
                    onValueChange={(value) => field.onChange(parseInt(value, 10))}
                  >
                    <FormControl>
                      <SelectTrigger className="w-50">
                        <SelectValue placeholder={t('Select a duration')} />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {durationOptions.map((opt) => (
                        <SelectItem key={opt.value} value={String(opt.value)}>
                          {opt.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <div className="flex gap-4">
              <Button
                type="button"
                onClick={() => {
                  if (capture.status === 'in_progress') {
                    handleStopCapture();
                  } else {
                    form.handleSubmit(handleStartCapture)();
                  }
                }}
                disabled={capture.status === 'pending' || (capture.status === 'in_progress' && !capture.pid)}
              >
                {capture.status === 'pending'
                  ? t('Starting Capture...')
                  : capture.status === 'in_progress'
                    ? t('Stop Capture')
                    : t('Start Capture')}
              </Button>
            </div>
          </div>
        </form>
      </Form>
      {isLoading ? (
        <Card className="border-border bg-background relative h-full min-h-[280px] w-full space-y-4 rounded-xl border p-4 shadow-lg md:p-5">
          <div className="mb-2 flex items-center gap-3">
            <Skeleton className="h-7 w-40 rounded-md" />
            <div className="ml-auto flex items-center gap-1">
              <Skeleton className="h-9 w-9 rounded-md" />
              <Skeleton className="h-9 w-9 rounded-md" />
            </div>
          </div>
          <div className="grid grid-cols-1 gap-x-6 gap-y-8 text-sm sm:grid-cols-2 lg:grid-cols-3">
            {Array.from({ length: 6 }).map((_, i) => (
              <Skeleton key={i} className="h-[174px] w-full rounded-xl" />
            ))}
          </div>
        </Card>
      ) : (
        capture.id && (
          <Card className="border-border bg-background relative h-full min-h-[280px] w-full space-y-4 rounded-xl border p-4 shadow-lg md:p-5">
            <div className="mb-2 flex items-center gap-3">
              <h1 className="text-foreground text-2xl font-bold tracking-tight">{t('Latest Capture :')}</h1>
              <div className="text-2xl font-semibold">
                <Duration
                  started_at={capture.started_at as string | undefined}
                  finished_at={capture.finished_at as string | undefined}
                />
              </div>
              <div className="ml-auto flex items-center gap-1">
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="secondary"
                      aria-label={t('View Capture')}
                      onClick={handleView}
                      disabled={!capture.id || capture.status === 'in_progress' || capture.status === 'pending'}
                    >
                      <Eye className="h-5 w-5" />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>{t('View Capture')}</TooltipContent>
                </Tooltip>
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="secondary"
                      aria-label={t('Download Capture')}
                      onClick={() => handleDownload(capture.id ?? 0)}
                      disabled={!capture.id || capture.status === 'in_progress' || capture.status === 'pending'}
                    >
                      <Download className="h-5 w-5" />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>{t('Download Capture')}</TooltipContent>
                </Tooltip>
              </div>
            </div>
            <div className="grid grid-cols-1 gap-x-6 gap-y-8 text-sm sm:grid-cols-2 lg:grid-cols-3">
              <InfoCard
                header={t('Interface')}
                data={capture.interface || ' '}
                icon={<Server />}
                tooltip={t('Network interface used for the capture.')}
              />
              <InfoCard
                header={t('Capture ID')}
                data={capture.id ?? ' '}
                icon={<Hash />}
                tooltip={t('Unique identifier for the capture.')}
              />
              <InfoCard
                header={t('Status')}
                data={
                  <Badge variant={variant} className="mt-1 rounded-md px-3 py-1 text-lg">
                    {status || ' '}
                  </Badge>
                }
                icon={
                  capture.status === 'in_progress' || capture.status === 'pending' ? (
                    <div className="flex items-center gap-2">
                      <LoaderCircle className="h-5 w-5 animate-spin" />
                    </div>
                  ) : capture.status === 'completed' ? (
                    <div className="flex items-center gap-2">
                      <CheckCircle className="h-5 w-5" />
                    </div>
                  ) : capture.status === 'failed' ? (
                    <div className="flex items-center gap-2">
                      <XCircle className="h-5 w-5" />
                    </div>
                  ) : (
                    <span>{t('Unknown')}</span>
                  )
                }
                tooltip={t('Current status of the capture.')}
              />
              <InfoCard
                header={t('Started')}
                data={formatDateTime(capture.started_at)}
                icon={<Clock />}
                tooltip={t('Timestamp when the capture started.')}
              />
              <InfoCard
                header={t('Finished')}
                data={capture.finished_at ? formatDateTime(capture.finished_at) : ' '}
                icon={<Clock />}
                tooltip={t('Timestamp when the capture finished.')}
              />
              <InfoCard
                header={t('Filename')}
                data={fileNameOnly || ' '}
                icon={<FileText />}
                tooltip={t('Name of the file where the capture is saved.')}
                cardContentClassName="break-all"
              />
            </div>
          </Card>
        )
      )}
    </div>
  );
}
