import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import { NetworkInterface, WiresharkCapture } from '@/components/wireshark/livecapture/types';

export function useGetNetworkInterfaces() {
  return useQuery<NetworkInterface[], AxiosError>({
    queryKey: ['NetworkInterfaces'],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/wireshark/interfaces`);
      return response.data.interfaces;
    },
  });
}

export function useGetLatestCapture() {
  return useQuery<WiresharkCapture | null, AxiosError>({
    queryKey: ['activecapture'],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/wireshark/latest-capture`);
      return response.data.capture ?? null;
    },
    staleTime: 10000,
  });
}

export function useStartCapture() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: ({ networkInterface, duration }: { networkInterface: string; duration: number }) =>
      axios.post(`/api/v1/wireshark/captures`, { interface: networkInterface, duration: duration }),
    onSuccess: () => {
      toast.success(t('Capture started.'));
      queryClient.invalidateQueries({ queryKey: ['activecapture'] });
    },
    onError: () => {
      toast.error(t('Failed to start capture.'));
    },
  });
}

export function useStopCapture() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (processId: number) => axios.post(`/api/v1/wireshark/captures/${processId}/stop`),
    onSuccess: () => {
      toast.success(t('Capture stopped.'));
      queryClient.invalidateQueries({ queryKey: ['activecapture'] });
    },
    onError: (error: AxiosError) => {
      toast.error(t('Failed to stop capture.'), {
        description: error.message,
      });
    },
  });
}
