import { Bar, BarChart, CartesianGrid, XAxis, Cell } from 'recharts';
import React from 'react';
import { useTranslation } from 'react-i18next';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { ChartConfig, ChartContainer, ChartTooltip, ChartTooltipContent } from '@/components/ui/chart';
import { useCapturesOverTime } from '@/components/wireshark/overview/queries';
import { DailyCaptureCount } from '@/components/wireshark/overview/types';
import { Skeleton } from '@/components/ui/skeleton';
import { cn } from '@/lib/utils';
import { useNavigate } from 'react-router';

const baseChartConfig = {
  total_captures: {
    label: 'Total Captures',
  },
} satisfies ChartConfig;

export function CapturesOverTimeChart({ className }: { className?: string }) {
  const { t } = useTranslation();
  const { data: capturesData, isLoading } = useCapturesOverTime();
  const navigate = useNavigate();

  const chartData: (DailyCaptureCount & { fill: string })[] = React.useMemo(() => {
    if (!capturesData) return [];
    return capturesData.captures_over_time.map((item, index) => ({
      ...item,
      fill: `var(--chart-${(index % 10) + 1})`,
    }));
  }, [capturesData]);

  const dynamicChartConfig: ChartConfig = React.useMemo(() => {
    return {
      ...baseChartConfig,
      ...chartData.reduce<Record<string, { label: string; color: string }>>((acc, item) => {
        acc[item.date] = { label: item.date, color: item.fill };
        return acc;
      }, {}),
    };
  }, [chartData]);

  const handleBarClick = (barData: DailyCaptureCount) => {
    navigate(`/wireshark?tab=history&filter=${encodeURIComponent(barData.date)}`);
  };

  if (isLoading) {
    return (
      <Card className={cn('flex flex-col', className)}>
        <CardHeader className="items-center pb-0">
          <Skeleton className="mb-2 h-6 w-48" />
          <Skeleton className="h-4 w-72" />
        </CardHeader>
        <CardContent className="flex-1 pb-0">
          <div className="mx-auto flex aspect-auto h-[200px] w-full items-center justify-center">
            <Skeleton className="h-full w-full rounded-md" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className={cn('flex flex-col', className)}>
      <CardHeader className="items-center pb-0">
        <CardTitle>{t('Captures Over Last 7 Days')}</CardTitle>
        <CardDescription>{t('Completed captures over the past week')}</CardDescription>
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        <ChartContainer config={dynamicChartConfig} className="mx-auto aspect-auto h-[200px] w-full">
          <BarChart data={chartData} margin={{ top: 10, right: 10, left: 10, bottom: 10 }}>
            <CartesianGrid vertical={false} strokeDasharray="3 3" />
            <XAxis
              dataKey="date"
              tickLine={false}
              axisLine={false}
              tickMargin={8}
              angle={-45}
              textAnchor="end"
              height={60}
            />
            <ChartTooltip content={<ChartTooltipContent className="w-[150px]" nameKey="total_captures" />} />
            <Bar className="cursor-pointer" dataKey="total_captures" radius={[4, 4, 0, 0]} onClick={handleBarClick}>
              {chartData.map((entry) => (
                <Cell key={entry.date} fill={entry.fill} />
              ))}
            </Bar>
          </BarChart>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col gap-2 text-sm" />
    </Card>
  );
}
