import { Pie, PieChart, Cell } from 'recharts';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from '@/components/ui/chart';
import { cn } from '@/lib/utils';
import { useTranslation } from 'react-i18next';
import { useSearchParams } from 'react-router';

interface PieChartProps {
  title: string;
  description?: string;
  data: Record<string, number>;
  className?: string;
  allowClick: boolean;
}

export function OverviewPieChart({ title, description, data, className, allowClick }: PieChartProps) {
  const { t } = useTranslation();
  const [_, setSearchParams] = useSearchParams();

  const chartData = Object.entries(data)
    .filter(([_, value]) => value > 0)
    .map(([label, value], index) => ({
      label,
      value,
      fill: `var(--chart-${(index % 10) + 1})`,
    }));

  const handleCellClick = (label: string) => {
    setSearchParams({ tab: 'history', filter: label });
  };

  const chartConfig: ChartConfig = {
    value: { label: 'Value' },
    ...Object.fromEntries(chartData.map((item) => [item.label, { label: item.label, color: item.fill }])),
  } satisfies ChartConfig;

  const hasData = chartData.length > 0;

  return (
    <Card className={cn('flex flex-col', className)}>
      <CardHeader className="items-center pb-0">
        <CardTitle>{t(title)}</CardTitle>
        {description && <CardDescription>{t(description)}</CardDescription>}
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        {hasData ? (
          <ChartContainer config={chartConfig} className="mx-auto aspect-square max-h-[250px]">
            <PieChart>
              <ChartTooltip cursor={false} content={<ChartTooltipContent hideLabel />} />
              <Pie data={chartData} dataKey="value" nameKey="label">
                {chartData.map((item) => (
                  <Cell
                    className={allowClick ? 'cursor-pointer' : ''}
                    key={item.label}
                    fill={item.fill}
                    onClick={allowClick ? () => handleCellClick(item.label) : () => {}}
                  />
                ))}
              </Pie>
              <ChartLegend
                content={<ChartLegendContent nameKey="label" />}
                className="-translate-y-2 flex-wrap gap-2 *:basis-1 *:justify-center"
              />
            </PieChart>
          </ChartContainer>
        ) : (
          <div className="text-muted-foreground mx-auto flex aspect-square max-h-[250px] items-center justify-center">
            {t('No data available')}
          </div>
        )}
      </CardContent>
    </Card>
  );
}
