import { useEffect, useState } from 'react';
import { columns } from '@/components/wireshark/captures/columns';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import { PacketDetailSheet } from '@/components/wireshark/PacketDetailSheet';
import { Packet } from '@/components/wireshark/captures/types';
import { useViewPackets } from '@/components/wireshark/ringbuffer/queries';
import {
  useReactTable,
  getCoreRowModel,
  getSortedRowModel,
  getPaginationRowModel,
  type SortingState,
} from '@tanstack/react-table';
import { useTranslation } from 'react-i18next';

interface DataTableProps {
  bufferId: number;
  initialFileName?: string;
  initialFilter?: string;
}

export default function RingBufferDataTable({ bufferId, initialFileName, initialFilter }: DataTableProps) {
  const [sheetOpen, setSheetOpen] = useState(false);
  const [selectedFrameNumber, setSelectedFrameNumber] = useState<number | null>(null);
  const [filename, setFilename] = useState<string>('');
  const [pageIndex, setPageIndex] = useState(0);
  const [pageSize, setPageSize] = useState(10);
  const [sorting, setSorting] = useState<SortingState>([{ id: 'frameNumber', desc: false }]);
  const [filter, setFilter] = useState<string>(initialFilter ?? '');
  const [lastTotal, setLastTotal] = useState<number>(0);
  const { t } = useTranslation();

  useEffect(() => {
    setFilter(initialFilter ?? '');
  }, [initialFilter]);

  const { data, isLoading, error } = useViewPackets({
    id: bufferId,
    file_name: initialFileName || '',
    pageIndex,
    pageSize,
    sortBy: sorting[0]?.id || 'frameNumber',
    sortOrder: sorting[0]?.desc ? 'desc' : 'asc',
    filter,
  });

  useEffect(() => {
    if (data?.total) {
      setLastTotal(data.total);
    }
  }, [data?.total]);

  const effectiveLoading = isLoading || bufferId === 0 || !initialFileName;

  useEffect(() => {
    if (data?.file_name) {
      setFilename(data.file_name);
    }
  }, [data?.file_name]);

  const handleRowClick = (row: Packet) => {
    setSelectedFrameNumber(row.frameNumber);
    setSheetOpen(true);
  };

  const handleGlobalFilterChange = (value: string) => {
    setPageIndex(0);
    setFilter(value);
  };

  const table = useReactTable<Packet>({
    data: data?.packets ?? [],
    columns: columns(t),
    state: {
      sorting,
      pagination: { pageIndex, pageSize },
    },
    manualSorting: true,
    manualPagination: true,
    pageCount: lastTotal ? Math.ceil(lastTotal / pageSize) : 1,
    onSortingChange: (updater) => {
      const next = typeof updater === 'function' ? updater(sorting) : updater;
      setSorting(next);
    },
    onPaginationChange: (updater) => {
      const next = typeof updater === 'function' ? updater({ pageIndex, pageSize }) : updater;
      setPageIndex(next.pageIndex);
      setPageSize(next.pageSize);
    },
    getCoreRowModel: getCoreRowModel(),
    getSortedRowModel: getSortedRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
  });

  if (error) {
    return (
      <div>
        {t('Error')}: {(error as Error).message}
      </div>
    );
  }

  return (
    <>
      <DataTable
        table={table}
        onRowClick={handleRowClick}
        onGlobalFilterChange={handleGlobalFilterChange}
        isLoading={effectiveLoading}
        defaultGlobalFilter={filter}
      />
      {selectedFrameNumber !== null && (
        <PacketDetailSheet
          isOpen={sheetOpen}
          onOpenChange={setSheetOpen}
          framenumber={selectedFrameNumber}
          filename={filename}
        />
      )}
    </>
  );
}
