import * as React from 'react';
import { useNavigate } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';

type Props = {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  src?: string;
  dst?: string;
  srcPort?: number;
  dstPort?: number;
  protocol?: string;
  interface?: string;
};

export default function SearchRingBufferDialog({
  open,
  onOpenChange,
  src,
  dst,
  srcPort,
  dstPort,
  protocol,
  interface: netInterface,
}: Props) {
  const { t } = useTranslation();
  const navigate = useNavigate();

  const [selectedFields, setSelectedFields] = React.useState<Record<string, boolean>>({
    src: false,
    dst: false,
    srcPort: false,
    dstPort: false,
    protocol: false,
    interface: false,
  });

  React.useEffect(() => {
    if (!open) return;
    setSelectedFields({
      src: false,
      dst: false,
      srcPort: false,
      dstPort: false,
      protocol: false,
      interface: false,
    });
  }, [open]);

  const Query = React.useMemo(() => {
    const parts: string[] = [];
    if (selectedFields.src && src) parts.push(`ip.src == ${src}`);
    if (selectedFields.dst && dst) parts.push(`ip.dst == ${dst}`);
    if (selectedFields.srcPort && srcPort !== undefined) parts.push(`tcp.srcport == ${srcPort}`);
    if (selectedFields.dstPort && dstPort !== undefined) parts.push(`tcp.dstport == ${dstPort}`);
    if (selectedFields.protocol && protocol) parts.push(`protocol == ${protocol}`);
    if (selectedFields.interface && netInterface) parts.push(`interface == "${netInterface}"`);
    return parts.join(' && ');
  }, [selectedFields, src, dst, srcPort, dstPort, protocol, netInterface]);

  const canSubmitQuery = !!Query?.trim();

  const toggleField = (key: string) => {
    setSelectedFields((prev) => ({ ...prev, [key]: !prev[key] }));
  };

  const onSubmit = () => {
    if (!canSubmitQuery) return;
    navigate(`/wireshark/ringbuffer?filter=${encodeURIComponent(Query)}`);
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent onClick={(e) => e.stopPropagation()}>
        <DialogHeader>
          <DialogTitle>{t('Search Wireshark Ring Buffer')}</DialogTitle>
        </DialogHeader>

        <div className="space-y-4 pt-4">
          <div className="min-h-[83px] space-y-2">
            <Label htmlFor="queryPreview">{t('Query preview')}</Label>
            <Input
              id="queryPreview"
              value={Query}
              readOnly
              aria-readonly
              placeholder={t('Select at least one field')}
              disabled
            />
            <p className="text-muted-foreground text-xs">
              {t('This uses Wireshark display filters, e.g.,')} <code>ip.src</code>, <code>ip.dst</code>,{' '}
              <code>tcp.srcport</code>.
            </p>
          </div>

          <div className="grid gap-2">
            {[
              { key: 'src', label: t('Use Source IP'), value: src },
              { key: 'dst', label: t('Use Destination IP'), value: dst },
              { key: 'srcPort', label: t('Use Source Port'), value: srcPort },
              { key: 'dstPort', label: t('Use Destination Port'), value: dstPort },
              { key: 'protocol', label: t('Use Protocol'), value: protocol },
              { key: 'interface', label: t('Use Interface'), value: netInterface },
            ]
              .filter((field) => field.value !== undefined)
              .map((field) => (
                <div
                  key={field.key}
                  onClick={() => toggleField(field.key)}
                  className={`flex cursor-pointer items-center justify-between rounded-md border px-3 py-2 shadow-sm transition ${
                    selectedFields[field.key] ? 'bg-primary/10 border-primary' : 'bg-background hover:bg-muted'
                  }`}
                >
                  <div className="flex flex-col">
                    <span className="text-sm font-medium">{field.label}</span>
                    <span className="text-muted-foreground text-xs">{String(field.value)}</span>
                  </div>
                </div>
              ))}
          </div>
        </div>

        <DialogFooter>
          <Button variant="secondary" onClick={() => onOpenChange(false)}>
            {t('Cancel')}
          </Button>
          <Button onClick={onSubmit} disabled={!canSubmitQuery}>
            {t('Search')}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
