import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { toast } from 'sonner';
import { RingBuffer } from '@/components/wireshark/ringbuffer/types';
import { useTranslation } from 'react-i18next';
import { Packet } from '@/components/wireshark/captures/types';

type ErrorResponse = {
  success?: boolean;
  message?: string;
};

export function useGetRingBuffer(id: number) {
  return useQuery<RingBuffer, AxiosError>({
    queryKey: ['ringbuffer', id],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/wireshark/ring-buffers/${id}`);
      return response.data;
    },
    retry: 1,
  });
}

export function useCreateRingBuffer() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (payload: {
      interfaces: string[];
      number_of_files: number;
      file_size: number;
      directory: string;
      file_name: string;
    }) => axios.post('/api/v1/wireshark/ring-buffers', payload),
    onSuccess: () => {
      toast.success(t('Ring Buffer Created.'));
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      toast.error(error.response?.data?.message || error.message);
    },
  });
}

export function useStopRingBuffer() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (ringbufferid: number) => axios.post(`/api/v1/wireshark/ring-buffers/${ringbufferid}/stop`),
    onSuccess: () => {
      toast.success(t('Ring Buffer Stopped.'));
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      toast.error(error.response?.data?.message || error.message);
    },
  });
}

export function useEditRingBuffer() {
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (payload: {
      id: number;
      interfaces: string[];
      number_of_files: number;
      file_size: number;
      directory: string;
      file_name: string;
    }) =>
      axios.put(`/api/v1/wireshark/ring-buffers/${payload.id}`, {
        interfaces: payload.interfaces,
        number_of_files: payload.number_of_files,
        file_size: payload.file_size,
        directory: payload.directory,
        file_name: payload.file_name,
      }),
    onSuccess: () => {
      toast.success(t('Ring Buffer Updated.'));
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      toast.error(error.response?.data?.message || error.message);
    },
  });
}

export function useStartRingBuffer() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation({
    mutationFn: (ringbufferid: number) => axios.post(`/api/v1/wireshark/ring-buffers/${ringbufferid}/start`),
    onSuccess: () => {
      toast.success(t('Ring Buffer Started.'));
      queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      toast.error(error.response?.data?.message || error.message);
    },
  });
}

export interface PacketResponse {
  file_name: string;
  packets: Packet[];
  total: number;
}

export interface UsePacketParams {
  id: number;
  file_name: string;
  pageIndex: number;
  pageSize: number;
  sortBy: string;
  sortOrder: 'asc' | 'desc';
  filter?: string;
}

export function useViewPackets({ id, file_name, pageIndex, pageSize, sortBy, sortOrder, filter }: UsePacketParams) {
  return useQuery<PacketResponse, AxiosError>({
    queryKey: ['wiresharkCapture', id, file_name, pageIndex, pageSize, sortBy, sortOrder, filter],
    queryFn: async () => {
      const response = await axios.get<PacketResponse>(`/api/v1/wireshark/ring-buffers/${id}/packets`, {
        params: {
          file_name,
          page: pageIndex + 1,
          per_page: pageSize,
          sortBy,
          sortOrder,
          ...(filter ? { filter } : {}),
        },
      });
      return response.data;
    },
    enabled: !!id && !!file_name,
  });
}

export interface MergedPcapResponse {
  message: string;
  id: number;
  file_path: string;
  start_time: string | null;
  end_time: string | null;
}

export function useGetMergedPcap(id: number) {
  return useQuery<MergedPcapResponse, AxiosError>({
    queryKey: ['wiresharkMergedPcap', id],
    queryFn: async () => {
      const response = await axios.get<MergedPcapResponse>(`/api/v1/wireshark/ring-buffers/${id}/merged-pcap`);
      return response.data;
    },
    enabled: !!id,
  });
}

export function useDownload() {
  const { t } = useTranslation();
  return useMutation({
    mutationFn: async (id: number) => {
      const response = await axios.get(`/api/v1/wireshark/ring-buffers/${id}/download`, { responseType: 'blob' });
      return { data: response.data, fileName: 'ringbuffer.pcap' };
    },
    onSuccess: ({ data, fileName }) => {
      const url = window.URL.createObjectURL(new Blob([data]));
      const link = document.createElement('a');
      link.href = url;
      link.setAttribute('download', fileName);
      document.body.appendChild(link);
      link.click();
      link.remove();
      toast.success(t('Ring Buffer Downloaded.'));
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      toast.error(error.response?.data?.message || error.message);
    },
  });
}
