import { createContext, useContext } from 'react';
import axios from '@/lib/axios';
import { useQuery } from '@tanstack/react-query';

export type Environment = {
  reverbAppKey: string;
  sessionLifetime: number;
  distro: string;
};

export type EnvironmentProviderState = {
  environment: Environment | null;
  isEnvironmentLoading: boolean;
};

const EnvironmentContext = createContext<EnvironmentProviderState | undefined>(undefined);

export function EnvironmentProvider({ children, ...props }: { children: React.ReactNode }) {
  const { data: environment, isLoading: isEnvironmentLoading } = useQuery({
    queryKey: ['environment'],
    queryFn: async () => {
      try {
        const response = await axios.get('/api/v1/environment');
        return response.data;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchOnWindowFocus: false,
  });

  return (
    <EnvironmentContext.Provider {...props} value={{ environment, isEnvironmentLoading }}>
      {children}
    </EnvironmentContext.Provider>
  );
}

export const useEnvironment = () => {
  const context = useContext(EnvironmentContext);
  if (context === undefined) {
    throw new Error('useEnvironment must be used within an EnvironmentProvider');
  }
  return context;
};
