import { ReactNode, useEffect } from 'react';
import i18n from 'i18next';
import { I18nextProvider, initReactI18next } from 'react-i18next';
import { useGetLanguage } from '@/components/queries';
import { useAuth } from '@/contexts/AuthContext';

import bg_BG from '../../lang/json/bg_BG.json';
import cs_CZ from '../../lang/json/cs_CZ.json';
import de_DE from '../../lang/json/de_DE.json';
import en_US from '../../lang/json/en_US.json';
import es_ES from '../../lang/json/es_ES.json';
import fr_FR from '../../lang/json/fr_FR.json';
import it_IT from '../../lang/json/it_IT.json';
import ja_JP from '../../lang/json/ja_JP.json';
import ko_KR from '../../lang/json/ko_KR.json';
import pl_PL from '../../lang/json/pl_PL.json';
import pt_PT from '../../lang/json/pt_PT.json';
import ru_RU from '../../lang/json/ru_RU.json';
import zh_CN from '../../lang/json/zh_CN.json';
import zh_TW from '../../lang/json/zh_TW.json';

const language_resources = {
  bg_BG: { translation: bg_BG },
  cs_CZ: { translation: cs_CZ },
  de_DE: { translation: de_DE },
  en_US: { translation: en_US },
  es_ES: { translation: es_ES },
  fr_FR: { translation: fr_FR },
  it_IT: { translation: it_IT },
  ja_JP: { translation: ja_JP },
  ko_KR: { translation: ko_KR },
  pl_PL: { translation: pl_PL },
  pt_PT: { translation: pt_PT },
  ru_RU: { translation: ru_RU },
  zh_CN: { translation: zh_CN },
  zh_TW: { translation: zh_TW },
};

i18n.use(initReactI18next).init({
  resources: language_resources,
  lng: 'en_US',
  fallbackLng: 'en_US',
  interpolation: {
    escapeValue: false,
  },
  returnEmptyString: false,
});

export const LanguageProvider = ({ children }: { children: ReactNode }) => {
  const { user } = useAuth();
  const { data: language, refetch: refetchLanguage } = useGetLanguage({ id: user?.id });

  useEffect(() => {
    if (user) {
      refetchLanguage();
    }
  }, [user, refetchLanguage]);

  useEffect(() => {
    // if the language data is available, use it
    if (language) {
      i18n.changeLanguage(language);
      return;
    }

    // if the language data is not available, use the local language
    const local_lang = localStorage.getItem('locale');
    if (local_lang) {
      i18n.changeLanguage(local_lang);
      return;
    }
  }, [language]);

  return <I18nextProvider i18n={i18n}>{children}</I18nextProvider>;
};
