import { PageLayout, PageHeader, PageTitle, PageDescription, PageSeparator } from '@/components/Page';
import { useForm } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { Button } from '@/components/ui/button';
import { TimeZoneDropdown } from '@/components/administration/global/TimeZoneDropdown';
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from '@/components/ui/command';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { Check, ChevronsUpDown } from 'lucide-react';
import { Card, CardContent } from '@/components/ui/card';
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel } from '@/components/ui/form';
import { useGetConfigOptions, useUpdateConfigOptions } from '@/components/administration/queries';
import { ConfigOptions } from '@/components/administration/types';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Language } from '@/components/user/types';
import { cn } from '@/lib/utils';
import { useState } from 'react';

export default function Globals() {
  const { t } = useTranslation();
  const updateConfig = useUpdateConfigOptions();
  const { data: configData } = useGetConfigOptions();
  const [open, setOpen] = useState(false);

  const form = useForm<ConfigOptions>({
    values: {
      theme: (configData && configData.theme) ?? 'dark',
      lang: (configData && configData.lang) ?? 'en_US',
      timezone: (configData && configData.timezone) ?? '(UTC-06:00) Central Time (US & Canada)',
    },
  });

  const { isDirty } = form.formState;

  const languages: { label: string; value: Language }[] = [
    { label: t('Bulgarian'), value: 'bg_BG' },
    { label: t('Czech'), value: 'cs_CZ' },
    { label: t('German'), value: 'de_DE' },
    { label: t('English'), value: 'en_US' },
    { label: t('Spanish'), value: 'es_ES' },
    { label: t('French'), value: 'fr_FR' },
    { label: t('Italian'), value: 'it_IT' },
    { label: t('Japanese'), value: 'ja_JP' },
    { label: t('Korean'), value: 'ko_KR' },
    { label: t('Polish'), value: 'pl_PL' },
    { label: t('Portugese'), value: 'pt_PT' },
    { label: t('Russian'), value: 'ru_RU' },
    { label: t('Chinese (China)'), value: 'zh_CN' },
    { label: t('Chinese (Taiwan)'), value: 'zh_TW' },
  ];

  const onSubmit = (values: ConfigOptions) => {
    updateConfig.mutate(values);
  };

  return (
    <PageLayout>
      <PageHeader>
        <PageTitle>{t('Global Settings')}</PageTitle>
        <PageDescription>{t('Manage settings for your entire Network Analyzer application')}</PageDescription>
        <PageSeparator />
      </PageHeader>
      <Card>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)}>
              <div className="flex flex-row items-center justify-between">
                <h1 className="text-md font-large leading-9 font-semibold">{t('Global Settings')}</h1>
                <Button disabled={!isDirty} variant={isDirty ? 'default' : 'outline'} className="border" type="submit">
                  {t('Update')}
                </Button>
              </div>
              <PageSeparator />
              {/* Language Dropdown */}
              <FormField
                control={form.control}
                name="lang"
                render={({ field }) => (
                  <FormItem className="flex flex-wrap items-center justify-between gap-4">
                    <div className="space-y-1">
                      <FormLabel>{t('Default Language')}</FormLabel>
                      <FormDescription>
                        {t('Change the default language users will see unless they set their own')}
                      </FormDescription>
                    </div>
                    <Popover open={open} onOpenChange={setOpen}>
                      <PopoverTrigger asChild>
                        <FormControl>
                          <Button
                            variant="outline"
                            role="combobox"
                            className={cn('w-[200px] justify-between', !field.value && 'text-muted-foreground')}
                          >
                            {field.value
                              ? languages.find((lang) => lang.value === field.value)?.label
                              : 'Select language'}
                            <ChevronsUpDown className="opacity-50" />
                          </Button>
                        </FormControl>
                      </PopoverTrigger>
                      <PopoverContent className="w-[200px] p-0">
                        <Command>
                          <CommandInput placeholder="Search language..." />
                          <CommandList className="custom-scrollbar">
                            <CommandEmpty>{t('No language found')}</CommandEmpty>
                            <CommandGroup>
                              {languages.map((language) => (
                                <CommandItem
                                  value={language.label}
                                  key={language.value}
                                  onSelect={() => {
                                    setOpen(false);
                                    form.setValue('lang', language.value, { shouldDirty: true });
                                  }}
                                >
                                  {language.label}
                                  <Check
                                    className={cn(
                                      'ml-auto',
                                      language.value === field.value ? 'opacity-100' : 'opacity-0'
                                    )}
                                  />
                                </CommandItem>
                              ))}
                            </CommandGroup>
                          </CommandList>
                        </Command>
                      </PopoverContent>
                    </Popover>
                  </FormItem>
                )}
              />
              <PageSeparator />
              {/* Theme Dropdown */}
              <FormField
                control={form.control}
                name="theme"
                render={({ field }) => (
                  <FormItem className="flex flex-wrap items-center justify-between gap-4">
                    <div className="space-y-1">
                      <FormLabel>{t('Default Theme')}</FormLabel>
                      <FormDescription>
                        {t('The default theme all users will see until they set their own')}
                      </FormDescription>
                    </div>
                    <FormControl>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <SelectTrigger className="w-[200px]">
                          <SelectValue placeholder={t('Select theme')} />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="light">{t('Light')}</SelectItem>
                          <SelectItem value="dark">{t('Dark')}</SelectItem>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                )}
              />
              <PageSeparator />
              {/* Timezone */}
              <FormField
                control={form.control}
                name="timezone"
                render={({ field }) => (
                  <FormItem className="flex flex-wrap items-center justify-between gap-4">
                    <div className="space-y-1">
                      <FormLabel>{t('Timezone')}</FormLabel>
                      <FormDescription>{t('Set the timezone on this server.')}</FormDescription>
                    </div>
                    <FormControl>
                      <div className="w-md">
                        <TimeZoneDropdown {...field} />
                      </div>
                    </FormControl>
                  </FormItem>
                )}
              />
            </form>
          </Form>
        </CardContent>
      </Card>
    </PageLayout>
  );
}
