import { useEffect, useState } from 'react';
import { useLocation } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { type TFunction } from 'i18next';

import { GlobeLockIcon, NetworkIcon, RadarIcon, HardDrive } from 'lucide-react';

import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { PageDescription, PageHeader, PageLayout, PageSeparator, PageTitle } from '@/components/Page';
import { useGetIntegrations } from '@/components/queries';
import AddFlowSourceCheckModal from '@/components/alerting/modals/flow-source-checks/AddFlowSourceCheck';
import AddNmapCheckModal from '@/components/alerting/modals/nmap-checks/AddNmapCheck';

import AlertingFlowSources from '@/pages/alerting/child-pages/AlertingFlowSources';
import AlertingResponseSettings from '@/pages/alerting/child-pages/AlertingResponseSettings';
import AlertingNmap from '@/pages/alerting/child-pages/AlertingNmap';
import AlertingSuricata from '@/pages/alerting/child-pages/AlertingSuricata';
import AlertingSystem from '@/pages/alerting/child-pages/AlertingSystem';
import { SourceGroup } from '@/components/sourcegroups/types';
import { Source } from '@/components/source/types';
import { useGetSources } from '@/components/source/queries';
import { useGetSourceGroups } from '@/components/sourcegroups/queries';
import AddSuricataCheckModal from '@/components/alerting/modals/suricata-checks/AddSuricataCheck';
import AddSystemCheckModal from '@/components/alerting/modals/system-checks/AddSystemCheck';

export default function Alerting() {
  const { t } = useTranslation();
  const location = useLocation();
  const [activeTab, setActiveTab] = useState<string>('flow-sources');

  const { data: installedIntegrations, isLoading: installedIntegrationsLoading } = useGetIntegrations();
  const { data: sources } = useGetSources();
  const { data: sourcegroups } = useGetSourceGroups();

  useEffect(() => {
    const tab = location.hash.replace('#', '');
    if (tab && ['flow-sources', 'nmap', 'suricata', 'system'].includes(tab)) {
      setActiveTab(tab);
    }
  }, [location.hash]);

  useEffect(() => {
    // update the url hash to match the active tab
    if (activeTab && window.location.hash !== `#${activeTab}`) {
      history.pushState(null, '', `#${activeTab}`);
    }
  }, [activeTab]);

  return (
    <PageLayout>
      <PageHeader>
        <div className="flex items-start justify-between">
          <div className="space-y-4">
            <PageTitle>{t('Alerting')}</PageTitle>
            <PageDescription>{t('Define checks that can alert users or trigger actions.')}</PageDescription>
          </div>
          <AlertingResponseSettings />
        </div>
        <PageSeparator className="mb-0 border-transparent" />
      </PageHeader>

      <Tabs defaultValue="flow-sources" onValueChange={(value) => setActiveTab(value)} value={activeTab}>
        <div className="flex justify-between">
          <TabsList className="mb-4 w-min">
            <TabsTrigger value="flow-sources" className="flex items-center gap-1.5">
              <NetworkIcon size={16} />
              {t('Flow Sources')}
            </TabsTrigger>
            {installedIntegrations?.nmap && (
              <TabsTrigger
                value="nmap"
                className="flex items-center gap-1.5"
                disabled={installedIntegrationsLoading || !installedIntegrations?.nmap}
              >
                <RadarIcon size={16} />
                {'Nmap'}
              </TabsTrigger>
            )}
            {installedIntegrations?.suricata && (
              <TabsTrigger
                value="suricata"
                className="flex items-center gap-1.5"
                disabled={installedIntegrationsLoading || !installedIntegrations?.suricata}
              >
                <GlobeLockIcon size={16} />
                {'Suricata'}
              </TabsTrigger>
            )}
            <TabsTrigger value="system" className="flex items-center gap-1.5">
              <HardDrive size={16} />
              {t('System')}
            </TabsTrigger>
          </TabsList>
          <AddNewButton t={t} activeTab={activeTab} sources={sources || []} sourcegroups={sourcegroups || []} />
        </div>

        <TabsContent value="flow-sources">
          <AlertingFlowSources />
        </TabsContent>
        <TabsContent value="nmap">
          <AlertingNmap />
        </TabsContent>
        <TabsContent value="suricata">
          <AlertingSuricata />
        </TabsContent>
        <TabsContent value="system">
          <AlertingSystem />
        </TabsContent>
      </Tabs>
    </PageLayout>
  );
}

const AddNewButton = ({
  activeTab,
  sources,
  sourcegroups,
}: {
  t: TFunction;
  activeTab: string;
  sources: Source[];
  sourcegroups: SourceGroup[];
}) => {
  switch (activeTab) {
    case 'flow-sources':
      return <AddFlowSourceCheckModal sources={sources} sourcegroups={sourcegroups} />;
    case 'nmap':
      return <AddNmapCheckModal />;
    case 'suricata':
      return <AddSuricataCheckModal />;
    case 'system':
      return <AddSystemCheckModal />;
    default:
      return null;
  }
};
