import { Button } from '@/components/ui/button';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Row } from '@tanstack/react-table';
import { useTranslation } from 'react-i18next';
import {
  DropdownMenu,
  DropdownMenuItem,
  DropdownMenuContent,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  ActivityIcon,
  HardDriveIcon,
  Loader2Icon,
  MoreHorizontalIcon,
  SquareTerminalIcon,
  TargetIcon,
  Trash2Icon,
  Settings2Icon,
} from 'lucide-react';
import AddNagiosServerModal from '@/components/alerting/modals/settings/AddNagiosServer';
import AddServiceHostnamesModal from '@/components/alerting/modals/settings/AddServiceHostname';
import {
  useDeleteCommand,
  useDeleteServiceHostname,
  useDeleteNagiosServer,
  useGetCommands,
  useGetNagiosServers,
  useGetServiceHostnames,
  useGetSNMPReceivers,
  useDeleteSNMPReceiver,
} from '@/components/alerting/queries';
import AddSNMPReceiverModal from '@/components/alerting/modals/settings/AddSnmpReceiver';
import AddCommandModal from '@/components/alerting/modals/settings/AddCommand';
import { Command, NagiosServer, ServiceHostname, SNMPReceiver } from '@/components/alerting/types';
import { useState } from 'react';
import EditCommandModal from '@/components/alerting/modals/settings/EditCommand';
import EditNagiosServerModal from '@/components/alerting/modals/settings/EditNagiosServer';
import EditServiceHostnameModal from '@/components/alerting/modals/settings/EditServiceHostname';
import EditSNMPReceiverModal from '@/components/alerting/modals/settings/EditSnmpReceiver';
import { type TFunction } from 'i18next';
import { useAuth } from '@/contexts/AuthContext';

import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle, SheetTrigger } from '@/components/ui/sheet';
import { isAdmin } from '@/components/role/utils';

export default function AlertingResponseSettings() {
  const { t } = useTranslation();
  const [sheetOpen, setSheetOpen] = useState(false);

  return (
    <div>
      <Sheet open={sheetOpen} onOpenChange={setSheetOpen}>
        <SheetTrigger asChild>
          <Button variant={'outline'} className="w-fit">
            <Settings2Icon size={16} />
            {t('Alerting Configs')}
          </Button>
        </SheetTrigger>
        <SheetContent className="custom-scrollbar overflow-y-auto sm:max-w-[800px]">
          <SheetHeader>
            <SheetTitle>{t(`Alerting Configs`)}</SheetTitle>
            <SheetDescription>{t(`Configure how alerts are handled and what actions are taken.`)}</SheetDescription>
          </SheetHeader>
          <div className="mt-4">
            <Tabs defaultValue="nagios-setup">
              <TabsList className="mb-4 w-min">
                <TabsTrigger value="nagios-setup" className="flex items-center gap-1.5">
                  <HardDriveIcon size={16} />
                  {t('Nagios Setup')}
                </TabsTrigger>
                <TabsTrigger value="snmp-receivers" className="flex items-center gap-1.5">
                  <TargetIcon size={16} />
                  {t('SNMP Receivers')}
                </TabsTrigger>
                <TabsTrigger value="commands" className="flex items-center gap-1.5">
                  <SquareTerminalIcon size={16} />
                  {t('Commands')}
                </TabsTrigger>
              </TabsList>

              <TabsContent value="nagios-setup">
                <AlertingNagiosSetup t={t} />
              </TabsContent>

              <TabsContent value="snmp-receivers">
                <AlertingSnmpReceivers t={t} />
              </TabsContent>

              <TabsContent value="commands">
                <AlertingCommands t={t} />
              </TabsContent>
            </Tabs>
          </div>
        </SheetContent>
      </Sheet>
    </div>
  );
}

function NagiosServerActionsCell({ t, nagiosServer }: { t: TFunction; nagiosServer: NagiosServer }) {
  const { user } = useAuth();
  const { mutate: deleteNagiosServer } = useDeleteNagiosServer();
  const [nagiosServerDropdownOpen, setNagiosServerDropdownOpen] = useState(false);

  return (
    <div className="flex justify-end">
      {isAdmin(user?.role) && (
        <DropdownMenu open={nagiosServerDropdownOpen} onOpenChange={setNagiosServerDropdownOpen}>
          <DropdownMenuTrigger asChild>
            <Button size="sm" variant="ghost" className="gap-2 text-xs">
              <MoreHorizontalIcon />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            <EditNagiosServerModal
              nagiosServer={nagiosServer}
              onModalClose={() => setNagiosServerDropdownOpen(false)}
            />
            <DropdownMenuItem onClick={() => deleteNagiosServer(nagiosServer.id)}>
              <Trash2Icon />
              <span className="hidden lg:block">{t(`Delete`)}</span>
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      )}
    </div>
  );
}

const AlertingNagiosSetup = ({ t }: { t: TFunction }) => {
  const { user } = useAuth();

  const { data: nagiosServers, isLoading: nagiosServersLoading } = useGetNagiosServers();
  const { data: serviceHostnames, isLoading: serviceHostnamesLoading } = useGetServiceHostnames();

  const { mutate: deleteServiceHostname } = useDeleteServiceHostname();

  const alertingNagiosSetupColumns = [
    {
      accessorKey: 'name',
      header: t('Name'),
    },
    {
      accessorKey: 'nrdp_url',
      header: t('NRDP URL'),
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => <div className="hidden" />,
      cell: ({ row }: { row: Row<NagiosServer> }) => <NagiosServerActionsCell t={t} nagiosServer={row.original} />,
    },
  ];

  function ServiceHostnameActionsCell({ t, serviceHostname }: { t: TFunction; serviceHostname: ServiceHostname }) {
    const [serviceHostnameDropdownOpen, setServiceHostnameDropdownOpen] = useState(false);

    return (
      <div className="flex justify-end">
        {isAdmin(user?.role) && (
          <DropdownMenu open={serviceHostnameDropdownOpen} onOpenChange={setServiceHostnameDropdownOpen}>
            <DropdownMenuTrigger asChild>
              <Button size="sm" variant="ghost" className="gap-2 text-xs">
                <MoreHorizontalIcon />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <EditServiceHostnameModal
                serviceHostname={
                  serviceHostnames?.find(
                    (serviceHostname: ServiceHostname) => serviceHostname.id == serviceHostname.id
                  ) as ServiceHostname
                }
                onModalClose={() => setServiceHostnameDropdownOpen(false)}
              />
              <DropdownMenuItem onClick={() => deleteServiceHostname(serviceHostname.id)}>
                <Trash2Icon />
                <span className="hidden lg:block">{t(`Delete`)}</span>
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        )}
      </div>
    );
  }

  const alertingServiceHostnamesColumns = [
    {
      accessorKey: 'servicename',
      header: t('Service Name'),
    },
    {
      accessorKey: 'hostname',
      header: t('Hostname'),
    },
    {
      accessorKey: 'nagios_server_id',
      header: t('Nagios Server'),
      cell: ({ row }: { row: Row<ServiceHostname> }) => (
        <div>
          <span className="text-xs">
            {nagiosServers?.find((nagiosServer: NagiosServer) => nagiosServer.id === row.original.server_id)?.name}
          </span>
        </div>
      ),
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => <div className="hidden" />,
      cell: ({ row }: { row: Row<ServiceHostname> }) => (
        <ServiceHostnameActionsCell t={t} serviceHostname={row.original} />
      ),
    },
  ];

  return (
    <div className="flex flex-col gap-12">
      {/* Nagios Servers Section */}
      <div className="flex flex-col gap-4">
        <div className="flex w-full justify-between gap-2">
          <div className="flex flex-col gap-1">
            <h1 className="flex items-center gap-1.5 text-xl font-medium">
              <HardDriveIcon size={20} />
              {t(`Nagios Servers`)}
            </h1>
            <h3 className="text-muted-foreground text-sm">
              {t(`Define the Nagios XI servers where alerts/notifications will be sent.`)}
            </h3>
          </div>
          <AddNagiosServerModal />
        </div>
        {nagiosServersLoading ? (
          <div className="flex h-48 justify-center">
            <Loader2Icon className="animate-spin" />
          </div>
        ) : (
          nagiosServers && (
            <DataTable columns={alertingNagiosSetupColumns} data={nagiosServers || []} hideView hidePageSizeSelector />
          )
        )}
      </div>
      {/* Service/Hostnames Section */}
      <div className="flex flex-col gap-4">
        <div className="flex w-full justify-between gap-2">
          <div className="flex flex-col gap-1">
            <h1 className="flex items-center gap-1.5 text-xl font-medium">
              <ActivityIcon size={20} />
              {t(`Service/Hostnames`)}
            </h1>
            <h3 className="text-muted-foreground text-sm">
              {t(
                `Define the service/hostnames sent to Nagios XI, appearing as unconfigured objects until configured in Nagios XI.`
              )}
            </h3>
          </div>
          <AddServiceHostnamesModal />
        </div>
        {serviceHostnamesLoading ? (
          <div className="flex h-48 items-center justify-center">
            <Loader2Icon className="animate-spin" />
          </div>
        ) : (
          serviceHostnames && (
            <DataTable columns={alertingServiceHostnamesColumns} data={serviceHostnames || []} hideView />
          )
        )}
      </div>
    </div>
  );
};

function SnmpReceiverActionsCell({ t, snmpReceiver }: { t: TFunction; snmpReceiver: SNMPReceiver }) {
  const { user } = useAuth();
  const [snmpReceiverDropdownOpen, setSnmpReceiverDropdownOpen] = useState(false);
  const { mutate: deleteSNMPReceiver } = useDeleteSNMPReceiver();

  return (
    <div className="flex justify-end">
      {isAdmin(user?.role) && (
        <DropdownMenu open={snmpReceiverDropdownOpen} onOpenChange={setSnmpReceiverDropdownOpen}>
          <DropdownMenuTrigger asChild>
            <Button size="sm" variant="ghost" className="gap-2 text-xs">
              <MoreHorizontalIcon />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            <EditSNMPReceiverModal
              snmp_receiver={snmpReceiver}
              onModalClose={() => setSnmpReceiverDropdownOpen(false)}
            />
            <DropdownMenuItem onClick={() => deleteSNMPReceiver(snmpReceiver.id)}>
              <Trash2Icon />
              <span className="hidden lg:block">{t(`Delete`)}</span>
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      )}
    </div>
  );
}

const AlertingSnmpReceivers = ({ t }: { t: TFunction }) => {
  const { data: snmp_receivers, isLoading: snmp_receivers_loading } = useGetSNMPReceivers();

  const alertingSnmpReceiversColumns = [
    {
      accessorKey: 'name',
      header: 'Name',
    },
    {
      accessorKey: 'ip',
      header: 'IP Address',
    },
    {
      accessorKey: 'port',
      header: 'Port',
    },
    {
      accessorKey: 'version',
      header: 'Version',
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => <div className="hidden" />,
      cell: ({ row }: { row: Row<SNMPReceiver> }) => <SnmpReceiverActionsCell t={t} snmpReceiver={row.original} />,
    },
  ];

  return (
    <div className="flex flex-col gap-4">
      <div className="flex w-full justify-between gap-2">
        <div className="flex flex-col gap-1">
          <h1 className="flex items-center gap-1.5 text-xl font-medium">
            <TargetIcon size={20} />
            {t(`SNMP Receivers`)}
          </h1>
          <h3 className="text-muted-foreground text-sm">
            {t(`Define the SNMP receivers where alerts/notifications will be sent.`)}
          </h3>
        </div>
        <AddSNMPReceiverModal />
      </div>
      {snmp_receivers_loading ? (
        <div className="flex h-48 items-center justify-center">
          <Loader2Icon className="animate-spin" />
        </div>
      ) : (
        snmp_receivers && <DataTable columns={alertingSnmpReceiversColumns} data={snmp_receivers || []} />
      )}
    </div>
  );
};

function CommandActionsCell({ t, command }: { t: TFunction; command: Command }) {
  const { user } = useAuth();
  const [dropdownOpen, setDrodownOpen] = useState(false);
  const { mutate: deleteCommand } = useDeleteCommand();

  return (
    <div className="flex justify-end">
      {isAdmin(user?.role) && (
        <DropdownMenu open={dropdownOpen} onOpenChange={setDrodownOpen}>
          <DropdownMenuTrigger asChild>
            <Button size="sm" variant="ghost" className="gap-2 text-xs">
              <MoreHorizontalIcon />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            <EditCommandModal command={command} onModalClose={() => setDrodownOpen(false)} />
            <DropdownMenuItem onClick={() => deleteCommand(command.id)}>
              <Trash2Icon />
              <span className="hidden lg:block">{t(`Delete`)}</span>
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      )}
    </div>
  );
}

const AlertingCommands = ({ t }: { t: TFunction }) => {
  const { data: commands, isLoading: commandsLoading } = useGetCommands();

  const alertingCommandsColumns = [
    {
      accessorKey: 'name',
      header: 'Name',
    },
    {
      accessorKey: 'location',
      header: 'Script Location',
    },
    {
      accessorKey: 'script',
      header: 'Script Name',
    },
    {
      accessorKey: 'arguments',
      header: 'Passed Args',
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => <div className="hidden" />,
      cell: ({ row }: { row: Row<Command> }) => <CommandActionsCell t={t} command={row.original} />,
    },
  ];

  return (
    <div className="flex flex-col gap-4">
      <div className="flex w-full justify-between gap-2">
        <div className="flex flex-col gap-1">
          <h1 className="flex items-center gap-1.5 text-xl font-medium">
            <SquareTerminalIcon size={20} />
            {t(`Commands`)}
          </h1>
          <h3 className="text-muted-foreground text-sm">{t(`Define custom scripts available for checks to use.`)}</h3>
        </div>

        <AddCommandModal />
      </div>
      {commandsLoading ? (
        <div className="flex h-48 items-center justify-center">
          <Loader2Icon className="animate-spin" />
        </div>
      ) : (
        commands && (
          <DataTable
            columns={alertingCommandsColumns}
            data={commands || []}
            defaultColumnVisibility={{
              arguments: false,
            }}
          />
        )
      )}
    </div>
  );
};
