import { useTranslation } from 'react-i18next';
import { Button } from '@/components/ui/button';
import { DataTable } from '@/components/ui/DataTable/DataTable';
import { ColumnDef } from '@tanstack/react-table';
import {
  DropdownMenu,
  DropdownMenuItem,
  DropdownMenuContent,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  ClipboardCheckIcon,
  Loader2Icon,
  MoreHorizontalIcon,
  PlayIcon,
  PowerIcon,
  PowerOffIcon,
  Trash2Icon,
} from 'lucide-react';
import { useDeleteCheck, useForceCheck, useGetChecks, useToggleActiveCheck } from '@/components/alerting/queries';
import { Badge } from '@/components/ui/badge';
import { SystemCheck, SystemCheckMetric } from '@/components/alerting/types';
import { useState } from 'react';
import { DataTableColumnHeader } from '@/components/ui/DataTable/DataTableColumnHeader';
import { Switch } from '@/components/ui/switch';
import EditSystemCheckModal from '@/components/alerting/modals/system-checks/EditSystemCheck';
import { useEcho } from '@laravel/echo-react';
import { useQueryClient } from '@tanstack/react-query';

export default function AlertingSystem() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const { data: checks, isLoading: checksLoading } = useGetChecks({ check_type: 'system' });
  const { mutate: deleteCheck } = useDeleteCheck();
  const { mutate: runCheck } = useForceCheck();
  const { mutate: toggleActiveCheck } = useToggleActiveCheck();

  useEcho(`NNA.Users`, 'CheckJobCompleted', () => {
    queryClient.invalidateQueries({ queryKey: ['checks', 'system'] });
  });

  function ActionsCell({ systemCheck }: { systemCheck: SystemCheck }) {
    const [dropdownOpen, setDrodownOpen] = useState(false);

    return (
      <div className="flex justify-end">
        <DropdownMenu open={dropdownOpen} onOpenChange={setDrodownOpen}>
          <DropdownMenuTrigger asChild>
            <Button size="sm" variant="ghost" className="gap-2 text-xs">
              <MoreHorizontalIcon />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            <DropdownMenuItem onClick={() => runCheck(systemCheck.id)}>
              <PlayIcon />
              <span className="hidden lg:block">{t(`Run Check`)}</span>
            </DropdownMenuItem>
            <EditSystemCheckModal
              check={checks.filter((check: SystemCheck) => check.id == systemCheck.id)[0]}
              onModalClose={() => setDrodownOpen(false)}
            />
            <DropdownMenuItem onClick={() => toggleActiveCheck(systemCheck.id)}>
              {systemCheck.active ? <PowerOffIcon /> : <PowerIcon />}
              <span className="hidden lg:block">{systemCheck.active ? t('Deactivate') : t('Activate')}</span>
            </DropdownMenuItem>
            <DropdownMenuItem onClick={() => deleteCheck(systemCheck.id)}>
              <Trash2Icon />
              <span className="hidden lg:block">{t(`Delete`)}</span>
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      </div>
    );
  }

  const alertingChecksColumns: ColumnDef<SystemCheck>[] = [
    {
      accessorKey: 'active',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Active" />;
      },
      cell: ({ row }) => {
        const value = row.getValue('active');

        return (
          <div className="flex px-2.5">
            <Switch checked={value as boolean} onCheckedChange={() => toggleActiveCheck(row.original.id)} />
          </div>
        );
      },
    },
    {
      accessorKey: 'name',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Name" />;
      },
    },
    {
      accessorKey: 'metric',
      header: t('Check Metric'),
      cell: ({ row }) => {
        const metricLabels: Record<SystemCheckMetric, string> = {
          cpu_usage: t('CPU Usage (%)'),
          used_percent: t('Memory Used (%)'),
          swap_percent: t('Memory Swap (%)'),
          available: t('Storage Available (bytes)'),
          percent: t('Storage Used (%)'),
        };

        const systemCheck = row.original as SystemCheck;

        return <div>{metricLabels[systemCheck.metric]}</div>;
      },
    },
    {
      accessorKey: 'last_code',
      header: () => {
        return <div className="w-min">{t('Last Status')}</div>;
      },
      cell: ({ row }) => {
        return <ReturnCodeStatus status_code={row.original.last_code} />;
      },
    },
    {
      accessorKey: 'last_run',
      header: t('Last Run'),
      cell: ({ row }) => {
        if (!row.original.last_run) {
          return <div>{`-`}</div>;
        }

        const datetime_formatted = new Date(`${row.original.last_run}`).toLocaleString();
        const datetime_array = datetime_formatted.split(',');
        return (
          <div className="flex h-min flex-col gap-0">
            <span>{`${datetime_array[0]},`}</span>
            <span>{`${datetime_array[1]}`}</span>
          </div>
        );
      },
    },
    {
      accessorKey: 'last_stdout',
      header: t('Last Stdout'),
      cell: ({ row }) => {
        if (!row.original.last_stdout) {
          return <div>{`-`}</div>;
        }

        return <div className="flex text-wrap">{row.original.last_stdout}</div>;
      },
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => <div className="hidden" />,
      cell: ({ row }) => <ActionsCell systemCheck={row.original} />,
    },
  ];

  return (
    <div className="flex flex-col gap-4">
      <h1 className="flex items-center gap-1.5 text-xl font-medium">
        <ClipboardCheckIcon size={20} />
        {t(`System Checks`)}
      </h1>
      {checksLoading ? (
        <div className="flex h-48 items-center justify-center">
          <Loader2Icon className="animate-spin" />
        </div>
      ) : (
        checks && <DataTable columns={alertingChecksColumns} data={checks || []} />
      )}
    </div>
  );
}

const ReturnCodeStatus = ({ status_code }: { status_code: string | undefined }) => {
  const status_code_dict: {
    [key: number]: {
      label: string;
      css: string;
      variant: 'green' | 'yellow' | 'red' | 'secondary';
    };
  } = {
    0: {
      label: 'OK',
      css: 'bg-success',
      variant: 'green',
    },
    1: {
      label: 'WARNING',
      css: 'bg-warning',
      variant: 'yellow',
    },
    2: {
      label: 'CRITICAL',
      css: 'bg-error',
      variant: 'red',
    },
    3: {
      label: 'UNKNOWN',
      css: 'bg-secondary',
      variant: 'secondary',
    },
  };

  if (status_code == undefined) {
    return <Badge variant={'secondary'}>{'Pending'}</Badge>;
  }

  const code = parseInt(status_code, 10);

  return (
    <Badge className="w-4/5 min-w-fit" variant={status_code_dict[code].variant}>
      {status_code_dict[code].label}
    </Badge>
  );
};
