import '/node_modules/react-grid-layout/css/styles.css';
import '/node_modules/react-resizable/css/styles.css';
import GridLayout, { WidthProvider, Responsive, Layout } from 'react-grid-layout';
import Dashlet from '@/components/dashboard/Dashlet';
import { useEditDashboard } from '@/components/dashboard/queries';
import { useGetDashlets } from '@/components/dashboard/dashlet/queries';
import { useParams, useSearchParams } from 'react-router-dom';
import { dashlet } from '@/components/dashboard/types';
import { Suspense, useCallback, useRef } from 'react';
import DashboardNotFound from '@/pages/dashboard/DashboardNotFound';
import { DashboardProvider } from '@/components/dashboard/DashboardContext';
import DashboardToolbar from '@/components/dashboard/toolbar/DashboardToolbar';
import { Skeleton } from '@/components/ui/skeleton';
import { ArrowDownRight } from 'lucide-react';
import AddDashletPlaceholder from '@/components/dashboard/AddDashletPlaceholder';
import React from 'react';

const ResponsiveReactGridLayout = WidthProvider(Responsive);

const CustomHandle = React.forwardRef<SVGSVGElement, { handleAxis: string }>((props, ref) => {
  const { handleAxis, ...restProps } = props;
  return (
    <ArrowDownRight
      ref={ref}
      {...restProps}
      className={`handle-${handleAxis} bg-background text-foreground absolute right-0 bottom-0 m-2 size-4 rounded opacity-0 transition-opacity group-hover/dashlet:opacity-100 hover:cursor-se-resize`}
    />
  );
});

export default function Dashboard() {
  const [searchParams] = useSearchParams();
  const isExporting = searchParams.get('export') === 'true';
  const { dashboardId } = useParams();
  const { data: dashlets, isLoading: isDashletsLoading, isError } = useGetDashlets(Number(dashboardId));
  const editDashboardMutation = useEditDashboard();
  const userInteractionRef = useRef(false);

  const onUserInteraction = () => (userInteractionRef.current = true);

  const areDashletsEqual = (a: dashlet[], b: dashlet[]): boolean => {
    if (a.length !== b.length) return false;

    return a.every((dashletA) => {
      const dashletB = b.find((d) => d.id === dashletA.id);
      if (!dashletB) return false;

      return (
        dashletA.x === dashletB.x &&
        dashletA.y === dashletB.y &&
        dashletA.width === dashletB.width &&
        dashletA.height === dashletB.height
      );
    });
  };

  const onLayoutChange = useCallback(
    (nextLayout: Layout[]) => {
      if (!userInteractionRef.current) {
        return;
      }

      if (!dashlets) return;
      const newDashlets = nextLayout
        .map((layoutItem) => {
          const oldDashlet = dashlets.find((d) => d.id === Number(layoutItem.i));
          if (oldDashlet) {
            const newDashlet: dashlet = {
              ...oldDashlet,
              height: layoutItem.h,
              width: layoutItem.w,
              x: layoutItem.x,
              y: layoutItem.y,
            };
            return newDashlet;
          } else {
            return null;
          }
        })
        .filter((dashlet): dashlet is dashlet => dashlet !== null);

      if (dashlets && !areDashletsEqual(dashlets, newDashlets)) {
        editDashboardMutation.mutate({ id: Number(dashboardId), dashlets: newDashlets });
      }

      userInteractionRef.current = false;
    },
    [dashlets, dashboardId, editDashboardMutation]
  );

  if (isError) {
    return <DashboardNotFound />;
  }

  if (isExporting) {
    const root = window.document.documentElement;
    root.classList.remove('light', 'dark');
    root.classList.add('light');
    return (
      <DashboardProvider id={Number(dashboardId)}>
        <GridLayout className="layout" cols={90} width={1495} rowHeight={10}>
          {dashlets?.map((dashlet) => {
            return (
              <div
                key={dashlet.id}
                data-grid={{
                  x: Number(dashlet.x),
                  y: Number(dashlet.y),
                  w: Number(dashlet.width),
                  h: Number(dashlet.height),
                }}
              >
                <Dashlet dashlet={dashlet} isLoading={isDashletsLoading} isExporting />
              </div>
            );
          })}
        </GridLayout>
      </DashboardProvider>
    );
  }

  return (
    <DashboardProvider id={Number(dashboardId)}>
      <div className="bg-background relative h-full w-full">
        <DashboardToolbar />
        {dashlets && dashlets.length > 0 ? (
          <ResponsiveReactGridLayout
            breakpoints={{ lg: 1200 }}
            cols={{ lg: 90 }}
            rowHeight={10}
            draggableHandle=".draggable"
            onDragStart={() => (userInteractionRef.current = true)}
            onResizeStart={onUserInteraction}
            onLayoutChange={onLayoutChange}
            resizeHandle={<CustomHandle handleAxis="se" />}
          >
            {dashlets.map((dashlet) => (
              <div
                className="group/dashlet"
                key={dashlet.id}
                data-grid={{
                  x: Number(dashlet.x),
                  y: Number(dashlet.y),
                  w: Number(dashlet.width),
                  h: Number(dashlet.height),
                }}
              >
                <Suspense fallback={<Skeleton className="h-full w-full rounded-md" />}>
                  <Dashlet dashlet={dashlet} isLoading={isDashletsLoading} />
                </Suspense>
              </div>
            ))}
          </ResponsiveReactGridLayout>
        ) : (
          <div className="px-[calc(.5rem+2px)] pt-[calc(.5rem+4px)]">
            <AddDashletPlaceholder dashboardId={Number(dashboardId)} />
          </div>
        )}
      </div>
    </DashboardProvider>
  );
}
