import { useTranslation } from 'react-i18next';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Link } from 'react-router-dom';
import { useForm } from 'react-hook-form';
import { MailQuestionIcon, ArrowLeft, CircleAlert, LoaderCircle } from 'lucide-react';
import axios, { AxiosError } from '@/lib/axios';
import { useMutation } from '@tanstack/react-query';
import { toast } from 'sonner';

export default function ForgotPassword() {
  const { t } = useTranslation();
  const form = useForm({
    defaultValues: {
      username: '',
    },
  });

  const { mutate: forgotPassword, isPending } = useMutation({
    mutationFn: (values: { username: string }) => {
      return axios.post('/api/v1/forgot-password', values);
    },
    onSuccess: async (response) => {
      toast(t('Password reset link sent'), { description: response.data.message });
    },
    onError: (error: AxiosError<{ message: string }>) => {
      toast(
        <div>
          <div className="flex items-center gap-2">
            <CircleAlert color={'var(--background)'} fill={'var(--foreground)'} className="h-5 w-5" />
            {t('Failed to send password reset link')}:
          </div>
          <div className="text-muted-foreground mt-2 max-h-50 overflow-auto">
            {error.response?.data.message ?? t('Please try again')}
          </div>
        </div>
      );
    },
  });

  function onSubmit(values: { username: string }) {
    forgotPassword(values);
  }
  return (
    <div className="mx-auto flex h-screen w-full max-w-md flex-col items-center justify-center p-4">
      <MailQuestionIcon size="30px" />
      <h2 className="my-3 text-xl font-medium">{t('Forgot Password')}</h2>
      <Form {...form}>
        <form className="flex w-full flex-col gap-4" onSubmit={form.handleSubmit(onSubmit)}>
          <FormField
            control={form.control}
            name="username"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-sm font-medium">{t('Username')}</FormLabel>
                <FormControl>
                  <Input placeholder="nagiosadmin" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <Button className="bg-primary" type="submit">
            {isPending ? <LoaderCircle className="animate-spin" /> : t('Reset Password')}
          </Button>
          <Link
            className="text-primary group mx-auto flex w-fit flex-row items-center justify-center gap-4"
            to={'/login'}
          >
            <ArrowLeft className="transition-transform group-hover:-translate-x-2" />
            {t('Back to Login')}
          </Link>
        </form>
      </Form>
    </div>
  );
}
