import { useTranslation } from 'react-i18next';
import { useForm } from 'react-hook-form';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Link } from 'react-router';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { PasswordInput } from '@/components/ui/passwordInput';
import axios, { AxiosError } from '@/lib/axios';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { toast } from 'sonner';
import LanuageDropdown from '@/components/ui/LanguageDropdown';
import { GlobeIcon } from 'lucide-react';
import { useGetGlobalLanguage } from '@/components/queries';
import { Skeleton } from '@/components/ui/skeleton';
import { useNavigate } from 'react-router';

type Credentials = {
  username: string;
  password: string;
};

export default function Login() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const { data: globalLanguage } = useGetGlobalLanguage();
  const navigate = useNavigate();
  const form = useForm<Credentials>({
    defaultValues: {
      username: '',
      password: '',
    },
  });

  const loginMutation = useMutation({
    mutationFn: (data: Credentials) => {
      return axios.post('/api/v1/login', {
        username: data.username,
        password: data.password,
      });
    },
    onSuccess: async (response) => {
      if (response.status === 202 && response.data?.requires_password_reset) {
        toast.info(t('Legacy Password Detected'), {
          description: t('Passwords from the previous version of Nagios Network Analyzer must be reset.'),
        });
        navigate('/forgot_password');
        return;
      }
      await queryClient.invalidateQueries();
    },
    onError: (error: AxiosError<{ message: string }>) => {
      const message = error?.response?.data?.message || t('Invalid username or password.');
      toast.error(message, { description: t('Please try again') });
    },
  });

  function onSubmit(credentials: Credentials) {
    axios.get('/sanctum/csrf-cookie').then(() => {
      loginMutation.mutate(credentials);
    });
  }

  return (
    <div className="mx-auto flex h-full h-screen max-w-xl flex-col items-center justify-center gap-4 px-5">
      {globalLanguage ? (
        <LanuageDropdown
          className="absolute top-8 right-8 w-fit gap-2"
          defaultValue={globalLanguage?.lang || 'en_US'}
          onChange={(value: string) => {
            localStorage.setItem('login_locale', value);
            queryClient.invalidateQueries({ queryKey: ['language'] });
          }}
        >
          <GlobeIcon className="h-4 w-4" />
        </LanuageDropdown>
      ) : (
        <Skeleton className="absolute top-8 right-8 h-9 w-30" />
      )}

      <h2 className="block text-5xl">{t('Welcome')}</h2>
      <p className="block">{t('Sign in to Nagios Network Analyzer')}</p>
      <Form {...form}>
        <form className="flex w-full flex-col gap-4" onSubmit={form.handleSubmit(onSubmit)}>
          <FormField
            control={form.control}
            name="username"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-sm font-medium">{t('Username')}</FormLabel>
                <FormControl>
                  <Input placeholder="Username" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="password"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-sm font-medium">{t('Password')}</FormLabel>
                <FormControl>
                  <PasswordInput {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <Link className="text-primary hover:underline" to={'/forgot_password'}>
            {t('Forget your password')}?
          </Link>
          <Button className="bg-primary w-full" type="submit">
            {t('Login')}
          </Button>
        </form>
      </Form>
    </div>
  );
}
