import { useTranslation } from 'react-i18next';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { PasswordInput } from '@/components/ui/passwordInput';
import { Button } from '@/components/ui/button';
import { Link } from 'react-router-dom';
import { useForm } from 'react-hook-form';
import { Lock, ArrowLeft, CircleAlert, LoaderCircle } from 'lucide-react';
import { useParams, useSearchParams } from 'react-router-dom';
import { useMutation } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { toast } from 'sonner';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useNavigate } from 'react-router-dom';

export default function ResetPassword() {
  const { t } = useTranslation();
  const { token } = useParams();
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();

  const formSchema = z
    .object({
      password: z.string().min(6, t('Password must be at least 6 characters')),
      password_confirmation: z.string(),
    })
    .refine((data) => data.password === data.password_confirmation, {
      message: 'Passwords do not match',
      path: ['password_confirmation'],
    });

  const form = useForm({
    resolver: zodResolver(formSchema),
    defaultValues: {
      password: '',
      password_confirmation: '',
    },
  });

  const { mutate: resetPassword, isPending } = useMutation({
    mutationFn: (values: {
      password: string;
      password_confirmation: string;
      username: string | null;
      token?: string;
    }) => {
      return axios.post('/api/v1/reset-password', values);
    },
    onSuccess: async (response) => {
      toast(t('Password reset'), { description: response.data.message });
      navigate('/login');
    },
    onError: (error: AxiosError<{ message: string }>) => {
      toast(
        <div>
          <div className="flex items-center gap-2">
            <CircleAlert color={'var(--background)'} fill={'var(--foreground)'} className="h-5 w-5" />
            {t('Failed to reset password')}:
          </div>
          <div className="text-muted-foreground mt-2 max-h-50 overflow-auto">
            {error.response?.data.message ?? t('Please try again')}
          </div>
        </div>
      );
    },
  });

  const username = searchParams.get('username');

  function onSubmit(values: z.infer<typeof formSchema>) {
    resetPassword({ ...values, username, token });
  }

  return (
    <div className="mx-auto flex h-screen w-full max-w-md flex-col items-center justify-center p-4">
      <Lock size="30px" />
      <h2 className="my-3 text-xl font-medium">{t('Reset Password')}</h2>
      <Form {...form}>
        <form className="flex w-full flex-col gap-4" onSubmit={form.handleSubmit(onSubmit)}>
          <FormField
            control={form.control}
            name="password"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-sm font-medium">{t('New Password')}</FormLabel>
                <FormControl>
                  <PasswordInput {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="password_confirmation"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-sm font-medium">{t('Confirm Password')}</FormLabel>
                <FormControl>
                  <PasswordInput {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <Button className="bg-primary" type="submit">
            {isPending ? <LoaderCircle className="animate-spin" /> : t('Submit')}
          </Button>
          <Link
            className="text-primary group mx-auto flex w-fit flex-row items-center justify-center gap-4"
            to={'/login'}
          >
            <ArrowLeft className="transition-transform group-hover:-translate-x-2" />
            {t('Back to Login')}
          </Link>
        </form>
      </Form>
    </div>
  );
}
