#!/bin/bash

# Checks if the script is run as root
if [ "$(id -u)" != "0" ]; then
    echo "This script must be run as root." 1>&2
    exit 1
fi

# Initialize distro and version variables
distro=""
version=""
arch=""

# Function to detect distro and version
detect_distro() {
    arch=$(uname -m)

    # Try /etc/os-release first (most reliable for modern systems)
    if [ -f /etc/os-release ]; then
        source /etc/os-release
        if [ -n "$ID" ]; then
            distro=$ID
            version=$VERSION_ID
            return 0
        fi
    fi

    # Fallback to /etc/lsb-release
    if [ -f /etc/lsb-release ]; then
        source /etc/lsb-release
        if [ -n "$DISTRIB_ID" ]; then
            distro=$(echo $DISTRIB_ID | tr '[:upper:]' '[:lower:]')
            version=$DISTRIB_RELEASE
            return 0
        fi
    fi

    # Fallback to checking specific release files
    if [ -f /etc/redhat-release ]; then
        if grep -qi "centos" /etc/redhat-release; then
            distro="centos"
            version=$(grep -oE '[0-9]+\.[0-9]+' /etc/redhat-release | head -1)
        elif grep -qi "red hat" /etc/redhat-release; then
            distro="rhel"
            version=$(grep -oE '[0-9]+\.[0-9]+' /etc/redhat-release | head -1)
        elif grep -qi "fedora" /etc/redhat-release; then
            distro="fedora"
            version=$(grep -oE '[0-9]+' /etc/redhat-release | head -1)
        fi
        return 0
    fi

    if [ -f /etc/debian_version ]; then
        distro="debian"
        version=$(cat /etc/debian_version)
        return 0
    fi

    # Additional fallbacks for other distributions
    if [ -f /etc/SuSE-release ]; then
        distro="opensuse"
        version=$(grep -oE '[0-9]+\.[0-9]+' /etc/SuSE-release | head -1)
        return 0
    fi

    return 1
}

# Detect the distribution
if detect_distro; then
    echo "Detected distro: $distro"
    echo "Detected version: $version"
    echo "Detected architecture: $arch"
else
    echo "Could not detect Linux distribution"
    exit 1
fi

# Checks to see if the script is run on Linux.
# Checks Linux Distro and obtains/installs prerequisites.
if [ "$(uname)" == "Linux" ]; then
    case $distro in
    "centos")
        echo "Installing dependencies for CentOS..."
        dnf config-manager --set-enabled crb
        dnf install -y wget tar libpcap-devel gcc flex byacc bison make bzip2
        ;;
    "rhel")
        echo "Installing dependencies for RHEL..."
        major_version=$(echo $version | cut -d. -f1)
        echo "Enabling codeready-builder-for-rhel-$major_version-$arch-rpms"
        subscription-manager repos --enable codeready-builder-for-rhel-$major_version-$arch-rpms
        dnf install -y wget tar libpcap-devel gcc flex byacc bison make bzip2
        ;;
    "fedora")
        echo "Installing dependencies for Fedora..."
        dnf install -y wget tar libpcap-devel gcc flex byacc bison make bzip2
        ;;
    "oracle" | "ol")
        echo "Installing dependencies for Oracle..."
        major_version=$(echo $version | cut -d. -f1)
        ol_codeready_builder="ol${major_version}_codeready_builder"
        echo "Enabling $ol_codeready_builder"
        dnf config-manager --set-enabled $ol_codeready_builder
        dnf install -y wget tar libpcap-devel gcc flex byacc bison make bzip2
        ;;
    "ubuntu" | "debian" | "linuxmint")
        echo "Installing dependencies for Debian-based system..."
        apt-get update
        apt-get install -y wget tar libpcap-dev gcc flex byacc bison make bzip2
        ;;
    "opensuse" | "suse" | "opensuse-leap" | "opensuse-tumbleweed")
        echo "Installing dependencies for openSUSE..."
        cd /tmp
        zypper install -n wget tar libpcap-devel gcc flex bison make bzip2
        wget ftp://ftp.pbone.net/mirror/carroll.cac.psu.edu/pub/linux/distributions/mandrakelinux/devel/cooker/i586/media/main/release/byacc-20121003-2-mdv2012.0.i586.rpm
        rpm -ivh byacc-20121003-2-mdv2012.0.i586.rpm
        ;;
    *)
        echo ""
        echo ""
        echo "Unknown/Unsupported Linux Distribution: $distro"
        echo ""
        exit 1
        ;;
    esac
fi

# Download and configure fprobe, also give basic instructions.
cd /tmp

echo "Downloading fprobe source code..."
if ! wget -O fprobe-1.1.tar.bz2 "http://downloads.sourceforge.net/project/fprobe/fprobe/1.1/fprobe-1.1.tar.bz2?r=http%3A%2F%2Fsourceforge.net%2Fprojects%2Ffprobe%2F&ts=1374682947&use_mirror=iweb"; then
    echo "ERROR: Failed to download fprobe source code"
    exit 1
fi

echo "Extracting fprobe source code..."
if ! tar xvf fprobe-1.1.tar.bz2; then
    echo "ERROR: Failed to extract fprobe source code"
    exit 1
fi

if ! cd fprobe-1.1; then
    echo "ERROR: Failed to enter fprobe source directory"
    exit 1
fi

echo "Configuring fprobe..."
if [ $distro == "opensuse" ] || [ $distro == "suse" ]; then
    if ! ./configure --exec-prefix=/usr; then
        echo "ERROR: Failed to configure fprobe for openSUSE/SUSE"
        exit 1
    fi
else
    if ! ./configure; then
        echo "ERROR: Failed to configure fprobe"
        exit 1
    fi
fi

echo "Compiling fprobe..."
if ! make; then
    echo "ERROR: Failed to compile fprobe"
    exit 1
fi

echo "Installing fprobe..."
if ! make install; then
    echo "ERROR: Failed to install fprobe"
    exit 1
fi

echo "Verifying fprobe installation..."
fp=$(which fprobe)
if [ -z "$fp" ]; then
    echo "ERROR: fprobe was not found in PATH after installation"
    exit 1
fi

echo ""
echo ""
echo ""
echo "SUCCESS: fprobe has been successfully installed at $fp"
echo ""
echo "To get started just enter:"
echo "$fp -i <interface> <NNA server>:<port>"
echo ""
echo ""

echo "To get fprobe to boot on restart enter the following command:"
echo "echo $fp -i <interface> <NNA server>:<port> >> /etc/rc.local"
echo ""
