#!/bin/bash
#
# Bash script that opens/closes firewall ports
# Copyright 2017-2025 Nagios Enterprises, LLC. All Rights reserved.
#
# Because different systems use different firewalls, we are going to guess
# what the firewall that is being used is (and if it's even running)
#
# Currently implemented:
#   IPT = iptables (CentOS/RHEL 5 & 6, sometimes CentOS/RHEL 7)
#   FWD = firewalld (CentOS/RHEL 7 standard)
#   UFW = uncomplicated firewall (Debian/Ubuntu)
#   NFT = nftables (Debian 10+, modern Linux)
#
# Needs implementation:
#   SUS = SuSEfirewall2 (most versions of SuSE/SLES)
#

FIREWALL="IPT"
FIREWALLSTATUS=0
PATH=$PATH:/sbin:/usr/sbin
BASEDIR=$(dirname $(readlink -f $0))

# Usage information
usage() {
    echo ""
    echo "Script that manages the configuration of the systems firewall, open and close ports,"
    echo "check the list of currently open ports/configurations."
    echo "Copyright 2017-2025 Nagios Enterprises, LLC. All Rights reserved."
    echo ""
    echo "-p | --port       The port to open"
    echo "-t | --type       Either 'tcp', 'udp', or 'both'"
    echo "--add             Add the port"
    echo "--rem             Remove the port"
    echo "--list            List the firewall rules"
    echo ""
    echo "-h | --help       Help information and usage documentation for this script"
    echo ""
}

# Get options passed
while [ -n "$1" ]; do
    case "$1" in
        --h | --help)
        usage
        exit 0
        ;;
        -p | --port)
        PORT=$2
        ;;
        -t | --type)
        TYPE=$2
        ;;
        --add)
        ACTION="add"
        ;;
        --rem)
        if [ "$ACTION" == "add" ]; then
            echo "You can only select either --add or --rem and not both"
            exit 0
        fi
        ACTION="rem"
        ;;
        --list)
        if [ -n "$ACTION" ]; then
            echo "You cannot use --add or --rem at the same time as --list"
            exit 0
        fi
        ACTION="list"
        ;;
    esac
    shift
done

# Skip validation checks if we're just listing
if [ "$ACTION" != "list" ]; then
    # Type must be tcp, udp, both
    if [ "$TYPE" != "tcp" ] && [ "$TYPE" != "udp" ] && [ "$TYPE" != "both" ]; then
        echo "You must specify tcp, udp, or both"
        exit 0
    fi

    # Port must be present
    if [ "x$PORT" == "x" ]; then
        echo "You must specify a port with -p | --port"
        exit 0
    fi
fi

# Check which firewall we are using
if [ `command -v firewall-cmd` ]; then
    FIREWALL="FWD"
elif [ `command -v ufw` ]; then
    FIREWALL="UFW"
elif [ `command -v nft` ]; then
    FIREWALL="NFT"
elif [ `command -v iptables` ]; then
    FIREWALL="IPT"
else
    echo "ERROR: No supported firewall found on this system."
    echo "Please install one of the following: firewalld, ufw, nftables, or iptables"
    exit 1
fi

echo "Detected firewall: $FIREWALL"

# Do iptables port adding and deleting
iptables_func() {
    iptables -C INPUT -p $2 --dport $PORT --jump ACCEPT > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        if [ "$1" == "I" ]; then
            iptables -$1 INPUT -p $2 -j ACCEPT --dport $PORT
        fi
    else
        if [ "$1" == "D" ]; then
            iptables -$1 INPUT -p $2 -j ACCEPT --dport $PORT
        fi
    fi
}

# Do firewalld port adding and deleting
firewalld_func() {
    # Add to runtime config (immediate effect)
    firewall-cmd --zone=public --$1=$PORT/$2
    # Also add to permanent config (survives reboot)
    firewall-cmd --zone=public --$1=$PORT/$2 --permanent
}

# Do ufw port adding and deleting
ufw_func() {
    if [ "$1" == "add" ]; then
        ufw allow $PORT/$2
    else
        ufw delete allow $PORT/$2
    fi
}

# Do nftables port adding and deleting
nftables_func() {
    # Check if input chain exists
    nft list chain inet filter input > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        # Create basic nftables structure if it doesn't exist
        nft add table inet filter > /dev/null 2>&1
        nft add chain inet filter input { type filter hook input priority 0 \; } > /dev/null 2>&1
    fi
    
    # Check if rule exists
    nft list chain inet filter input | grep -q "$2 dport $PORT accept"
    if [ $? -ne 0 ]; then
        if [ "$1" == "add" ]; then
            nft add rule inet filter input $2 dport $PORT accept
        fi
    else
        if [ "$1" == "remove" ]; then
            # Find and delete the rule handle
            HANDLE=$(nft -a list chain inet filter input | grep "$2 dport $PORT accept" | sed -n 's/.*# handle \([0-9]*\)/\1/p')
            if [ -n "$HANDLE" ]; then
                nft delete rule inet filter input handle $HANDLE
            fi
        fi
    fi
}

# Check what action we are taking
if [ "$ACTION" == "add" ]; then

    # Add firewall port
    case $FIREWALL in
        IPT)
            if [ "$TYPE" == "both" ]; then
                iptables_func "I" "udp"
                iptables_func "I" "tcp"
            else
                iptables_func "I" "$TYPE"
            fi
            service iptables save > /dev/null
        ;;
        FWD)
            if [ "$TYPE" == "both" ]; then
                firewalld_func "add-port" "udp"
                firewalld_func "add-port" "tcp"
            else
                firewalld_func "add-port" $TYPE
            fi
        ;;
        UFW)
            if [ "$TYPE" == "both" ]; then
                ufw_func "add" "udp"
                ufw_func "add" "tcp"
            else
                ufw_func "add" "$TYPE"
            fi
        ;;
        NFT)
            if [ "$TYPE" == "both" ]; then
                nftables_func "add" "udp"
                nftables_func "add" "tcp"
            else
                nftables_func "add" "$TYPE"
            fi
        ;;
    esac

elif [ "$ACTION" == "rem" ]; then

    # Remove firewall port
    case $FIREWALL in
        IPT)
            if [ "$TYPE" == "both" ]; then
                iptables_func "D" "udp"
                iptables_func "D" "tcp"
            else
                iptables_func "D" "$TYPE"
            fi
            service iptables save > /dev/null
        ;;
        FWD)
            if [ "$TYPE" == "both" ]; then
                firewalld_func "remove-port" "udp"
                firewalld_func "remove-port" "tcp"
            else
                firewalld_func "remove-port" $TYPE
            fi
        ;;
        UFW)
            if [ "$TYPE" == "both" ]; then
                ufw_func "remove" "udp"
                ufw_func "remove" "tcp"
            else
                ufw_func "remove" "$TYPE"
            fi
        ;;
        NFT)
            if [ "$TYPE" == "both" ]; then
                nftables_func "remove" "udp"
                nftables_func "remove" "tcp"
            else
                nftables_func "remove" "$TYPE"
            fi
        ;;
    esac

elif [ "$ACTION" == "list" ]; then

    # List the ports that are open and what they 
    echo "Listing firewall rules for: $FIREWALL"
    echo "================================"
    case $FIREWALL in
        IPT)
            echo "Open ports (iptables):"
            iptables -L INPUT -n -v | grep -E "ACCEPT.*dpt:" || echo "No explicit port rules found"
        ;;
        FWD)
            echo "Runtime configuration (currently active):"
            echo "Open ports:"
            firewall-cmd --list-ports
            echo "Active services:"
            firewall-cmd --list-services
            echo ""
            echo "Permanent configuration (survives reboot):"
            echo "Open ports:"
            firewall-cmd --permanent --list-ports
            echo "Active services:"
            firewall-cmd --permanent --list-services
        ;;
        UFW)
            echo "Firewall status and open ports (ufw):"
            ufw status numbered
        ;;
        NFT)
            echo "Open ports (nftables):"
            nft list ruleset | grep -E "(dport|sport)" || echo "No port rules found"
            echo ""
            echo "Full input chain:"
            nft list chain inet filter input 2>/dev/null || echo "No input chain configured"
        ;;
    esac
    echo "================================"
    exit 0

fi

echo "Firewall updated"
exit 0
