#!/bin/bash
#
# Bash script for restoring Nagios Network Analyzer
# Copyright 2014-2025 Nagios Enterprises, LLC. All Rights reserved.
#
# Restores a backup of Nagios Network Analyzer
#

# Make sure we have the required arguments
if [ $# != 2 ]; then
	echo "Usage: $0 <backup_file> <sql_dump_file>"
	echo "This script restores your Nagios Network Analyzer system using:"
	echo "  <backup_file>   - Previously made Nagios NA backup tar.gz file"
	echo "  <sql_dump_file>  - MySQL dump file created with: mysqldump --no-create-info --complete-insert nagiosna > /tmp/nagiosna.sql"
	echo "Copyright 2014-2025 Nagios Enterprises, LLC. All Rights reserved."
	exit 1
fi

# Get arguments
backup_file="$(realpath "$1")"
sql_dump_file="$(realpath "$2")"
ROOTDIR="/store/backups/nagiosna"
FILTER_SCRIPT="/usr/local/nagiosna/scripts/filter_sql_dump.py"

# Must be root
if [ "$(id -u)" -ne 0 ]; then
    echo "Error: You must be root to run this script." >&2
	exit 1
fi

##############################
# CHECK FOR FILTER SCRIPT
##############################
if [ ! -f "$FILTER_SCRIPT" ]; then
	echo "Error: SQL filter script not found at $FILTER_SCRIPT"
	echo "Make sure filter_sql_dump.py is in the same directory as this script."
	exit 1
fi

##############################
# MAKE SURE FILES EXIST
##############################
if [ ! -f "$backup_file" ]; then
	echo "Unable to find backup file $backup_file!"
	exit 1
fi

if [ ! -f "$sql_dump_file" ]; then
	echo "Unable to find SQL dump file $sql_dump_file!"
	echo ""
	echo "Please create an INSERT-only dump from your old system using:"
	echo "mysqldump --no-create-info --complete-insert nagiosna > /tmp/nagiosna.sql"
	exit 1
fi

##############################
# MAKE TEMP RESTORE DIRECTORY
##############################
ts=`date +%s`
echo "TS=$ts"
my_dir="${ROOTDIR}/${ts}-restore"
mkdir -p "$my_dir"
cp -f "$backup_file" "$my_dir"
if [ ! -d "$my_dir" ]; then
	echo "Unable to create restore directory $my_dir!"
	exit 1
fi

##############################
# UNZIP BACKUP
##############################
echo "Extracting backup to $my_dir..."
cd "$my_dir"
tar xzfps "$backup_file"

# Change to subdirectory
cd `ls | head -1`

# Make sure we have at least the flows tarball in here...
backup_dir=`pwd`
echo "In $backup_dir..."
if [ ! -f "$backup_dir/nagiosna-flows.tar.gz" ]; then
	echo "Unable to find flow files to restore in $backup_dir"
	exit 1
fi

echo "Backup files look okay.  Preparing to restore..."

##############################
# SHUTDOWN SERVICES
##############################
echo "Shutting down services..."
systemctl stop nagiosna

##############################
# RESTORE DIRS
##############################
flow_dir=/usr/local/nagiosna/var/

echo "Restoring flow files to ${flow_dir}..."
if [ ! -d "$flow_dir" ]; then
    mkdir -p "$flow_dir"
fi
tar xzfps "$backup_dir/nagiosna-flows.tar.gz" -C "$flow_dir"
tar_res=$?
if [ "$tar_res" -eq 2 ]; then
    echo "Error restoring flow files - tar returned $tar_res"
    exit 1
fi

##############################
# RESTORE DATABASES
##############################
echo "Restoring MySQL databases..."

set -a
source <(grep -E '^(DB_HOST|DB_USERNAME|DB_PASSWORD|DB_DATABASE)=' /var/www/html/nagiosna/.env)
set +a
export MYSQL_PWD="$DB_PASSWORD"

echo "Using SQL dump file: $sql_dump_file"

# Filter and transform the SQL dump
echo "Processing SQL dump file for migration..."
filtered_sql="${my_dir}/nagiosna_filtered.sql"

python3 "$FILTER_SCRIPT" "$sql_dump_file" "$filtered_sql"
filter_result=$?

if [ $filter_result != 0 ]; then
	echo "Error: Failed to filter SQL dump file!"
	echo "Check that filter_sql_dump.py is working correctly."
	exit 1
fi

if [ ! -f "$filtered_sql" ]; then
	echo "Error: Filtered SQL file was not created!"
	exit 1
fi

echo "Truncating selected tables in $DB_DATABASE before import..."

mysql -h "$DB_HOST" -u "$DB_USERNAME" "$DB_DATABASE" <<'EOF'
SET FOREIGN_KEY_CHECKS = 0;

TRUNCATE TABLE `commands`;
TRUNCATE TABLE `service_hostnames`;
TRUNCATE TABLE `nagios_servers`;
TRUNCATE TABLE `source_group_linker`;
TRUNCATE TABLE `source_groups`;
TRUNCATE TABLE `sources`;
TRUNCATE TABLE `snmp_receivers`;
TRUNCATE TABLE `auth_servers`;
TRUNCATE TABLE `checks`;
TRUNCATE TABLE `users`;
TRUNCATE TABLE `alerting_associations`;
TRUNCATE TABLE `views`;

SET FOREIGN_KEY_CHECKS = 1;
EOF

if [ $? -ne 0 ]; then
    echo "Error truncating tables — check MySQL credentials or table names."
    exit 1
fi

echo "Importing filtered database..."
mysql -h "$DB_HOST" -u "$DB_USERNAME" "$DB_DATABASE" < "$filtered_sql"
res=$?

if [ $res != 0 ]; then
	echo "Error restoring MySQL database '$DB_DATABASE' - check database credentials!"
	echo "Filtered SQL dump is available at: $filtered_sql"
	exit 1
fi

echo "Database restore completed successfully."

##############################
# UPDATE SOURCE DIRECTORIES
##############################
echo "Updating source directories to use new path format..."

mapfile -t sources < <(mysql -h "$DB_HOST" -u "$DB_USERNAME" \
    --batch --raw -N -e "SELECT CONCAT(id, '|', name, '|', directory) FROM sources;" "$DB_DATABASE")

for row in "${sources[@]}"; do
    IFS='|' read -r id name olddir <<< "$row"
    base_dir="$(dirname "$olddir")"
    [[ "$base_dir" != */ ]] && base_dir="${base_dir}/"
    newdir="${base_dir}${id}"

    if [ -d "$olddir" ] && [ ! -d "$newdir" ]; then
        echo "Renaming flow directory: '$olddir' -> '$newdir'"
        mv "$olddir" "$newdir"
    fi

    # Update DB
    mysql -h "$DB_HOST" -u "$DB_USERNAME" "$DB_DATABASE" \
        -e "UPDATE sources SET directory='${newdir}' WHERE id=${id};"
done

##############################
# RUN SEEDERS FOR ADDING ROLES TO USERS
##############################
echo "Updating user roles..."
cd /var/www/html/nagiosna && php artisan db:seed --class=AssignRolesToExistingUsersSeeder

##############################
# RUN SEEDERS FOR UPDATING API KEYS FOR MIGRATED USERS
##############################
echo "Updating API keys for migrated users..."
cd /var/www/html/nagiosna && php artisan db:seed --class=UpdateAPIKeysForMigratedUsersSeeder

##############################
# RESTART SERVICES
##############################
echo "Restarting services..."
systemctl start nagiosna

##############################
# CLEANUP
##############################
echo "Cleaning up temporary files..."

# Keep the filtered SQL for reference in case of issues
echo "Note: Filtered SQL dump saved at: $filtered_sql"
echo "You can safely delete $my_dir after verifying the restore."

# Optionally uncomment to auto-delete temp directory
# rm -rf $my_dir

echo " "
echo "==============="
echo "RESTORE COMPLETE"
echo "==============="
echo " "
echo "The following changes were made during migration:"
echo "- Removed obsolete tables (nagiosna_sessions, nagiosna_Reports, nagiosna_cmdsubsys, etc.)"
echo "- Renamed tables to new schema (e.g. nagiosna_Commands -> commands)"
echo "- Renamed columns to new schema (e.g. nrdp -> nrdp_url)"
echo " "

exit 0