#!/bin/bash

BACKUP_FILE="$1"

if [ -z "$BACKUP_FILE" ]; then
    echo "Please provide the path to the backup file"
    echo "Usage: $0 /path/to/external_flows.tar.gz"
    exit 1
fi

# Convert to absolute path if relative path provided
if [[ "$BACKUP_FILE" != /* ]]; then
    BACKUP_FILE="$(pwd)/$BACKUP_FILE"
fi

if [ ! -f "$BACKUP_FILE" ]; then
    echo "Backup file not found: $BACKUP_FILE"
    exit 1
fi

set -a
source <(grep -E '^(DB_HOST|DB_USERNAME|DB_PASSWORD|DB_DATABASE)=' /var/www/html/nagiosna/.env)
set +a
export MYSQL_PWD="$DB_PASSWORD"

# Create temporary directory for extraction
TEMP_RESTORE_DIR="/tmp/restore_$$"
mkdir -p "$TEMP_RESTORE_DIR"

# Extract the backup using absolute paths
echo "Extracting backup file: $BACKUP_FILE"
tar xzf "$BACKUP_FILE" -C "$TEMP_RESTORE_DIR"

# Process each sourceid directory in the flows directory
for source_dir in "$TEMP_RESTORE_DIR"/flows/sourceid_*; do
    if [ -d "$source_dir" ]; then
        # Extract the ID from the directory name
        dir_name=$(basename "$source_dir")
        id=$(echo "$dir_name" | grep -o '_[0-9]*_' | tr -d '_')
        
        if [ -n "$id" ]; then
            # Query the database to get the directory for this source ID
            target_dir=$(mysql -h "$DB_HOST" -u "$DB_USERNAME" -D "$DB_DATABASE" -sN -e \
                "SELECT directory FROM sources WHERE id = $id")

            if [ -z "$target_dir" ]; then
                echo "Warning: Source ID $id not found in database. Skipping..."
                continue
            fi

            echo "Restoring source ID $id to: $target_dir"

            # Check if target directory exists
            if [ ! -d "$target_dir" ]; then
                echo "Creating target directory: $target_dir"
                mkdir -p "$target_dir"
                chown nna:users "$target_dir"
                chmod 775 "$target_dir"
            fi
            
            echo "Restoring files for source ID $id..."
            
            # Restore flows directory using cp instead of rsync
            if [ -d "$source_dir/flows" ]; then
                echo "  Restoring flows..."
                cp -r "$source_dir/flows" "$target_dir/"
            fi
            
            # Restore views directory using cp
            if [ -d "$source_dir/views" ]; then
                echo "  Restoring views..."
                cp -r "$source_dir/views" "$target_dir/"
            fi
            
            # Restore RRD file
            if [ -f "$source_dir/bandwidth.rrd" ]; then
                echo "  Restoring bandwidth.rrd..."
                cp "$source_dir/bandwidth.rrd" "$target_dir/"
            fi
            
            chown -R nna:users "$target_dir"

            echo "Restore complete for source ID $id"
        else
            echo "Warning: Could not extract ID from directory name: $dir_name"
        fi
    fi
done

# Cleanup
echo "Cleaning up temporary files..."
rm -rf "$TEMP_RESTORE_DIR"

# Restart nagiosna service to reload sources
echo ""
echo "Restarting nagiosna service to activate restored sources..."
if systemctl is-active --quiet nagiosna; then
    systemctl restart nagiosna
    echo "nagiosna service restarted successfully"
else
    echo "Warning: nagiosna service is not running. Starting it now..."
    systemctl start nagiosna
fi

echo ""
echo "All sources have been restored and activated!"
echo "Restore process complete!"