<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\AlertingAssociation;
use App\Models\Command;
use App\Models\ServiceHostname;
use App\Models\User;
use App\Models\SnmpReceiver;

class Check extends Model
{
    public $timestamps = false;

    protected $fillable = [
        'active',
        'name',
        'object_type',
        'object_id',
        'metric',
        'warning',
        'critical',
        'raw_query',
        'last_val',
        'last_run',
        'last_code',
        'last_stdout',
        'check_type',
    ];

    public function alertingAssociations(): HasMany
    {
        return $this->hasMany(AlertingAssociation::class);
    }

    public function object()
    {
        return $this->morphTo(__FUNCTION__, 'object_type', 'object_id');
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'object_id' => 'integer',
            'warning' => 'string',
            'critical' => 'string',
        ];
    }

    /**
     * Get users associated with this check
     * 
     * @return \Illuminate\Support\Collection<int, User>
     */
    public function getUsers()
    {
        $association_ids = $this->alertingAssociations()
            ->where('association_type', 'user')
            ->pluck('association_id');

        return User::whereIn('id', $association_ids)->get();
    }

    /**
     * Get SNMP receivers associated with this check
     * 
     * @return \Illuminate\Support\Collection<int, SnmpReceiver>
     */
    public function getSnmpReceivers()
    {
        $association_ids = $this->alertingAssociations()
            ->where('association_type', 'snmp_receiver')
            ->pluck('association_id');

        return SnmpReceiver::whereIn('id', $association_ids)->get();
    }

    /**
     * Get commands associated with this check
     * 
     * @return \Illuminate\Support\Collection<int, Command>
     */
    public function getCommands()
    {
        $association_ids = $this->alertingAssociations()
            ->where('association_type', 'command')
            ->pluck('association_id');

        return Command::whereIn('id', $association_ids)->get();
    }

    /**
     * Get service hostnames (Nagios) associated with this check
     * 
     * @return \Illuminate\Support\Collection<int, ServiceHostname>
     */
    public function getServiceHostnames()
    {
        $association_ids = $this->alertingAssociations()
            ->where('association_type', 'nagios')
            ->pluck('association_id');

        return ServiceHostname::whereIn('id', $association_ids)->get();
    }
}
