<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Auth\Passwords\CanResetPassword as Resetable;
use Illuminate\Contracts\Auth\CanResetPassword;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Notifications\ResetPasswordUsername;

class User extends Authenticatable implements CanResetPassword
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable, Resetable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'username',
        'email',
        'password',
        'first_name',
        'last_name',
        'company',
        'phone',
        'lang',
        'type',
        'active',
        'created_on',
        'last_login',
        'apiaccess',
        'apikey',
        'apikey_id',
        'theme',
        'role_id',
        'auth_server_id',
        'auth_server_data',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'legacy_password' => 'boolean',
            'active' => 'boolean',
            'apiaccess' => 'boolean',
        ];
    }

    /**
     * The reports that belong to the user.
     */
    public function reports(): BelongsToMany
    {
        return $this->belongsToMany(Report::class);
    }

    /**
     * Get the role that owns the user.
     */
    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class);
    }

    /**
     * Determine if the user has the admin role.
     */
    public function isAdmin(): bool
    {
        return $this->role_id === 1;
    }

    public function hasPermission(string $crud, string $section, string $permission): bool
    {
        $this->loadMissing('role');

        return in_array($crud, $this->role->{$section}[$permission] ?? []);
    }

    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPasswordUsername($token));
    }
}
