<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\Facades\Schema;
use App\Models\NmapScheduledScan;
use App\Models\Source;
use App\Models\SourceGroup;
use App\Models\Report;
use App\Models\NmapScan;
use App\Models\ConfigOption;
use App\Models\SuricataConfig;
use App\Models\SuricataAlert;
use App\Models\SuricataData;
use App\Models\SuricataScan;
use App\Jobs\NmapScanJob;
use App\Jobs\ReportJob;
use App\Jobs\CheckJob;
use App\Events\NmapScanCreated;
use App\Models\Check;
use Cron\CronExpression;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Relations\Relation;
use App\Models\User;
use App\Models\NagiosServer;
use App\Models\SnmpReceiver;
use App\Models\Command;
use App\Services\ScheduleService;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Bus;


class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     * 
     * THIS FUNCTION MUST RETURN EARLY IF ANY DATABASE TABLE USED WITHIN DOES NOT EXIST!!! 
     * Laravel calls this function before migrations are ran! 
     */
    public function boot(): void
    {
        if (!Schema::hasTable('nagiosna.cache') ||
            !Schema::hasTable('nmap_scheduled_scans') ||
            !Schema::hasTable('suricata_rulesets') ||
            !Schema::hasTable('nagiosna.suricata_scan') || 
            !Schema::hasTable('reports') ||
            !Schema::hasTable('suricata_data'))
        {
            return;
        }
        // // if (app()->runningInConsole()) return;
        
        $timezone = Cache::remember('config.timezone', 60, function () {
            return ConfigOption::where('name', 'timezone')->value('value') ?? 'UTC';
        });

        Config::set('app.timezone', $timezone);
        Config::set('app.schedule_timezone', $timezone);
        date_default_timezone_set($timezone);

        Relation::morphMap([
            'user' => User::class,
            'nagios' => NagiosServer::class,
            'snmp_receiver' => SnmpReceiver::class,
            'command' => Command::class,
            'source' => Source::class,
            'sourcegroup' => SourceGroup::class,
            'scheduled_scan' => NmapScheduledScan::class,
        ]);

        // Define Gates for API routes/controllers that do not utilize a policy
        Gate::define('accessBackupController', function (User $user) {
            return $user->isAdmin();
        });

        Gate::define('accessWhois', function (User $user) {
            return true;
        });

        Gate::define('accessReverseDns', function (User $user) {
            return true;
        });

        Gate::define('createQuery', function (User $user) {
            return true;
        });

        Gate::define('getRules', function (User $user) {
            if ($user->isAdmin() || $user->hasPermission('get', 'suricata_permissions', 'rules')) {
                return true;
            }
            return false;
        });

        Gate::define('postRule', function (User $user) {
            if ($user->isAdmin() || $user->hasPermission('post', 'suricata_permissions', 'rules')) {
                return true;
            }
            return false;
        });

        Gate::define('putRule', function (User $user) {
            if ($user->isAdmin() || $user->hasPermission('put', 'suricata_permissions', 'rules')) {
                return true;
            }
            return false;
        });

        Gate::define('deleteRule', function (User $user) {
            if ($user->isAdmin() || $user->hasPermission('delete', 'suricata_permissions', 'rules')) {
                return true;
            }
            return false;
        });

        Gate::define('deleteRuleFile', function (User $user) {
            if ($user->isAdmin() || $user->hasPermission('delete', 'suricata_permissions', 'rules')) {
                return true;
            }
            return false;
        });

        $this->app->booted(function () {
            $schedule = app(Schedule::class);
            $scheduleService = app(ScheduleService::class);
            $timezone = ConfigOption::where('name', 'timezone')->value('value');

            // Schedule daily maintenance check
            $schedule->command('maintenance:check')->dailyAt('00:00')->name('Check Maintenance')->timezone($timezone);

            $suricataConfig = SuricataConfig::first();
            if ($suricataConfig && $suricataConfig->rulesets_seeded) {
                $schedule->command('rulesets:sync')->hourly()->name('Sync Rulesets')->timezone($timezone); // sync the sources file to the DB every hour
                if ($suricataConfig->auto_update_rules){
                    switch ($suricataConfig->ruleset_update_interval) {
                        case '5min':
                            $schedule->command('rulesets:update')->everyFiveMinutes()->name('Update Rulesets')->timezone($timezone);
                            break;
                        case 'hourly':
                            $schedule->command('rulesets:update')->hourly()->name('Update Rulesets')->timezone($timezone);
                            break;
                        case 'daily':
                            $schedule->command('rulesets:update')->daily()->name('Update Rulesets')->timezone($timezone);
                            break;
                        case 'weekly':
                            $schedule->command('rulesets:update')->weekly()->name('Update Rulesets')->timezone($timezone);
                            break;
                        case 'monthly':
                            $schedule->command('rulesets:update')->monthly()->name('Update Rulesets')->timezone($timezone);
                            break;
                        default:
                            Log::warning("Unknown ruleset update interval: {$suricataConfig->ruleset_update_interval}");
                            break;
                    }
                }
            }

            $suricataScan = SuricataScan::firstOrCreate(
                [],
                [
                    'pid' => null,
                    'running' => false,
                    'last_rotated' => null,
                    'started_at' => null,
                    'eve_current_line' => 0,
                    'current_eve_log' => null,
                ]
            );

            if ($suricataScan->isRunning()) {
                $schedule->command('suricata:read')->everyMinute()->name('Get suricata Alerts')->timezone($timezone);
            } 

            if (SuricataAlert::exists()){ 
                $schedule->command('alerts:suricata-rollover')->hourly()->name('Rotate alerts')->timezone($timezone);
            }
            if (SuricataData::exists()){
                $schedule->command('suricata:prune-data')->everyFiveMinutes()->name('Prune Suricata Data')->timezone($timezone);
            }

            $scheduledScans = NmapScheduledScan::where('enabled', true)->get();
            foreach ($scheduledScans as $scheduledScan) {
                $type = $scheduledScan->schedule_type;
                $params = $scheduledScan->schedule_parameters ?? [];

                // Use fields directly instead of building 'args'
                if ($type === 'cron' && !empty($params['expression'])) {
                    try {
                        CronExpression::factory($params['expression']);
                    } catch (\InvalidArgumentException $e) {
                        Log::warning("Invalid cron for scan ID {$scheduledScan->id}: {$e->getMessage()}");
                        continue;
                    }
                }

                $event = $schedule->call(function () use ($scheduledScan) {
                    $scan = NmapScan::create([
                        'user_id' => $scheduledScan->user_id,
                        'nmap_profile_id' => $scheduledScan->nmap_profile_id,
                        'scheduled_nmap_scan_id' => $scheduledScan->id,
                        'title' => $scheduledScan->name . ' [' . now()->format('Y-m-d H:i:s' . ']'),
                        'status' => 'Pending',
                        'scan_parameters' => $scheduledScan->parameters,
                    ]);

                    $associated_checks = Check::where('active', true)
                        ->where('object_type', 'scheduled_scan')
                        ->where('object_id', $scheduledScan->id)
                        ->get();

                    Bus::chain([
                        new NmapScanJob($scan),
                        ...$associated_checks->map(function ($check) {
                            return new CheckJob($check);
                        }),
                    ])->dispatch();
                    NmapScanCreated::dispatch($scan);

                    Log::info("Scheduled scan run", [
                        'scheduled_id' => $scheduledScan->id,
                        'scan_id' => $scan->id,
                    ]);
                })->name("Scheduled Nmap: {$scheduledScan->name}");

                if (!$scheduleService->applyTypeToEvent($type, $event, $params, $timezone)) {
                    continue;
                }
            }

            $reports = Report::where('enabled', true)->get();
            foreach ($reports as $report) {
                $type = $report->schedule;
                $params = $report->schedule_parameters ?? [];

                // Use fields directly instead of building 'args'
                if ($type === 'cron' && !empty($params['expression'])) {
                    try {
                        CronExpression::factory($params['expression']);
                    } catch (\InvalidArgumentException $e) {
                        Log::warning("Invalid cron for scan ID {$report->id}: {$e->getMessage()}");
                        continue;
                    }
                }

                $event = $schedule->call(function () use ($report) {
                    ReportJob::dispatch($report);
                })->name("Report: {$report->name}");

                if (!$scheduleService->applyTypeToEvent($type, $event, $params, $timezone)) {
                    continue;
                }
            }

            if (Schema::hasTable('checks')) {
                $active_checks = Check::where('active', true)->get();
                foreach ($active_checks as $check) {
                    $event = $schedule->call(function () use ($check) {
                        CheckJob::dispatch($check);

                        Log::debug("Scheduled check job", [
                            'check_id' => $check->id,
                            'check_name' => $check->name,
                        ]);
                    })->name("Scheduled Check: {$check->name}")->everyFiveMinutes();
                }
            }
        });
    }
}
