import React, { useState } from "react";
import {
    PageSeparator,
} from "@/components/Page";
import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
} from "@/components/ui/form";
import {
    Select,
    SelectContent,
    SelectGroup,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { SubmitHandler, useForm } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { usePostAuthServers, usePutAuthServers } from "@/components/administration/queries";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { AuthServer } from "@/components/administration/types"

export default function AddEditAuthServerDialog({ 
    AuthServer, 
    children,
    open,
    onOpenChange
}: { 
    AuthServer?: AuthServer;
    children?: React.ReactNode;
    open?: boolean;
    onOpenChange?: (open: boolean) => void;
}) {
    const [internalOpen, setInternalOpen] = useState(false);
    const { t } = useTranslation();
    const putAuthServer = usePutAuthServers();
    const postAuthServer = usePostAuthServers();

    // Use external state if provided, otherwise use internal state
    const isOpen = open !== undefined ? open : internalOpen;
    const handleOpenChange = onOpenChange || setInternalOpen;

    const form = useForm<AuthServer>({
        defaultValues: {
            id: AuthServer?.id ?? undefined,
            type: AuthServer?.type ?? "ad",
            active: AuthServer?.active ?? true,
            name: AuthServer?.name ?? "",
            host: AuthServer?.host ?? "",
            basedn: AuthServer?.basedn ?? "",
            suffix: AuthServer?.suffix ?? "",
            controllers: AuthServer?.controllers ?? "",
            encryption: AuthServer?.encryption ?? "ssl",
            port: AuthServer?.port ?? 389,
        },
        values: AuthServer,
    });

    // Watch the current selected server type
    const serverType = form.watch("type");

    const onSubmit: SubmitHandler<AuthServer> = (data) => {
        if (AuthServer){
            putAuthServer.mutate(data, {onSuccess: () => {
                handleOpenChange(false);
                form.reset();
                }
            });
        }
        else{
            postAuthServer.mutate(data, {onSuccess: () => {
                handleOpenChange(false);
                form.reset();
                }
            });
        }
    };
    
    const handleDialogChange = (isOpen: boolean) => {
        handleOpenChange(isOpen);
        if (!isOpen) {
            form.reset();
        }
    };

    const server = [
        { name: t("Active Directory"), value: "ad" },
        { name: "LDAP", value: "ldap" },
    ];
    
    const encryptionOptions = [
        { label: t("None"), value: "none" },
        { label: "SSL/TLS", value: "ssl" },
        { label: "STARTTLS", value: "tls" },
    ];

    return (
        <Dialog open={isOpen} onOpenChange={handleDialogChange}>
            {children && (
                <DialogTrigger asChild>
                    {children}
                </DialogTrigger>
            )}
            <DialogContent className="sm:max-w-[75vw] rounded-lg z-50">
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)}>
                        <DialogHeader>
                            <div className="flex justify-between">
                                <div className="flex md:w-2/3">
                                    <div>
                                    <DialogTitle className = "text-xl font-semibold px-6">
                                        {AuthServer ? t('Edit') : t('Add')} {t(" LDAP / Active Directory Server")}
                                    </DialogTitle>
                                    <DialogDescription className="px-6" >
                                        {t("You must make sure that you can access the LDAP / Active Directory server from your Network Analyzer box. You should also verify that the correct encryption methods are available. If you're planning on using SSL or TLS with self-signed certificates you need to make sure the proper certificates are installed on the Network Analyzer server or you will not be able to connect to your LDAP / Active Directory server.")}
                                    </DialogDescription>
                                    </div>  

                                </div>
                                <div className="flex justify-between px-6 space-x-6">
                                    {/* Active Checkbox */}
                                    <FormField
                                        control={form.control}
                                        name="active"
                                        render={({ field }) => (
                                            <FormItem className="flex items-start py-3 space-x-1">
                                                <FormControl>
                                                    <Checkbox
                                                    checked={!!field.value}
                                                    onChange={(e) => field.onChange((e.target as HTMLInputElement).checked)}
                                                    id="active-checkbox"
                                                    className="text-primary focus:ring-primary"
                                                    />
                                                </FormControl>
                                                <FormLabel>
                                                    {t("Active")}
                                                </FormLabel>
                                            </FormItem>
                                        )}
                                    />
                                    <Button type="submit" className="bg-primary">
                                        {AuthServer ? t('Update') : t('Add')} {t(" Server")}
                                    </Button>
                                </div>
                            </div>
                        </DialogHeader>
                        <div className="px-6">
                            <PageSeparator />

                            <FormField
                                control={form.control}
                                name="type"
                                render={({ field }) => (
                                <FormItem className="flex flex-col md:flex-row md:items-center">
                                    <div className="md:w-2/3">
                                        <FormLabel>{t("Server Type")}</FormLabel>
                                    </div>
                                    <FormControl>
                                        <Select
                                            value={field.value}
                                            onValueChange={(selectedVal) => {
                                            field.onChange(selectedVal)
                                        }}>
                                            <SelectTrigger className="w-full md:w-1/3">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                            <SelectGroup className="w-full max-h-80">
                                                {server.map(({name, value}) => (
                                                    <SelectItem key={value} value={value}>
                                                        {name}
                                                    </SelectItem>
                                                ))}
                                            </SelectGroup>
                                        </SelectContent>
                                        </Select>
                                    </FormControl>
                                </FormItem>
                                )}
                            />

                            <PageSeparator />
                            
                            <FormField
                                control={form.control}
                                name="name"
                                render={({ field }) => (
                                <FormItem className="flex">
                                    <div className="md:w-2/3 space-y-2">
                                    <FormLabel>{t("Server Name")}</FormLabel>
                                    <FormDescription>
                                        {t(
                                        "The name of the server for internal purposes only. This will not affect the connection."
                                        )}
                                    </FormDescription>
                                    </div>
                                    <div className="md:w-1/3">
                                    <FormControl>
                                        <Input {...field} placeholder={t("Server Name")} />
                                    </FormControl>
                                    </div>
                                </FormItem>
                                )}
                            />
                
                            <PageSeparator />

                            <FormField
                                control={form.control}
                                name="basedn"
                                render={({ field }) => (
                                <FormItem className="flex">
                                    <div className="md:w-2/3 space-y-2">
                                        <FormLabel>
                                            {t("Base DN")}
                                        </FormLabel>
                                        <FormDescription className="md:w-2/3">
                                            {t(
                                                "The LDAP-format starting object (distinguished name) that your users are defined below, such as DC=nagios,DC=com."
                                            )}
                                        </FormDescription>
                                    </div>
                                    <div className="md:w-1/3">
                                        <FormControl>
                                        <Input {...field} placeholder="DC=nagios, DC=com" />
                                        </FormControl>
                                    </div>
                                </FormItem>
                            )}
                            />

                            <PageSeparator />

                            {serverType === "ad" ? (
                                <>
                                    {/* Account Suffix */}
                                    <FormField
                                        control={form.control}
                                        name="suffix"
                                        render={({ field }) => (
                                        <FormItem className="flex">
                                            <div className="md:w-2/3 space-y-2">
                                            <FormLabel>{t("Account Suffix")}</FormLabel>
                                            <FormDescription>
                                                {t(
                                                "The part of the full user identification after the username, such as @nagios.com."
                                                )}
                                            </FormDescription>
                                            </div>
                                            <div className="md:w-1/3">
                                            <FormControl>
                                                <Input {...field} placeholder="@nagios.com" />
                                            </FormControl>
                                            </div>
                                        </FormItem>
                                        )}
                                    />

                                    <PageSeparator />

                                    <FormField
                                        control={form.control}
                                        name="controllers"
                                        render={({ field }) => (
                                        <FormItem className="flex">
                                            <div className="md:w-2/3 space-y-2">
                                            <FormLabel>{t("Domain Controllers")}</FormLabel>
                                            <FormDescription>
                                                {t("A comma-separated list of domain controllers.")}
                                            </FormDescription>
                                            </div>
                                            <div className="md:w-1/3">
                                            <FormControl>
                                                <Input
                                                {...field}
                                                placeholder="dc1.nagios.com,dc2.nagios.com"
                                                />
                                            </FormControl>
                                            </div>
                                        </FormItem>
                                        )}
                                    />
                                </>
                            ) : (
                                <>
                                    {/* LDAP Host */}
                                    <FormField
                                        control={form.control}
                                        name="host"
                                        render={({ field }) => (
                                            <FormItem className="flex">
                                                <div className="md:w-2/3 space-y-2">
                                                <FormLabel>{t("LDAP Host")}</FormLabel>
                                                <FormDescription>
                                                    {t("The hostname or IP address of your LDAP server.")}
                                                </FormDescription>
                                                </div>
                                                <div className="md:w-1/3">
                                                <FormControl>
                                                    <Input {...field} placeholder="ldap.nagios.com" />
                                                </FormControl>
                                                </div>
                                            </FormItem>
                                        )}
                                    />
                                    
                                    <PageSeparator />

                                    <FormField
                                        control={form.control}
                                        name="port"
                                        render={({ field }) => (
                                            <FormItem className="flex">
                                                <div className="md:w-2/3 space-y-2">
                                                    <FormLabel>{t("LDAP Port")}</FormLabel>
                                                    <FormDescription>
                                                        {t("The port number to connect to on the LDAP server. (Default = 389)")}
                                                    </FormDescription>
                                                    </div>
                                                    <div className="md:w-1/3">
                                                    <FormControl>
                                                        <Input {...field} type="number" placeholder="389" />
                                                    </FormControl>
                                                </div>
                                            </FormItem>
                                        )}
                                    />
                                </>
                            )}

                            <PageSeparator />
                            
                            <FormField
                                control={form.control}
                                name="encryption"
                                render={({ field }) => (
                                    <FormItem className="flex  md:items-center">
                                        <div className="md:w-2/3 space-y-2">
                                            <FormLabel className="md:w-2/3">
                                                {t("Encryption Method")}
                                            </FormLabel>
                                            <FormDescription className="md:w-2/3">
                                                {t(
                                                "The type of security (if any) to use for the connection to the server(s). The STARTTLS option may use a plain text connection if the server does not upgrade the connection to TLS"
                                                )}
                                            </FormDescription>
                                        </div>
                                        <FormControl>
                                            <Select
                                                value={field.value}
                                                onValueChange={(selectedVal) => {
                                                field.onChange(selectedVal)
                                            }}>
                                                <SelectTrigger className="w-full md:w-1/3">
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                <SelectGroup className="w-full max-h-80">
                                                    {encryptionOptions.map(({label, value}) => (
                                                        <SelectItem key={value} value={value}>
                                                            {label}
                                                        </SelectItem>
                                                    ))}
                                                </SelectGroup>
                                            </SelectContent>
                                            </Select>
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        </div>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
}