import {
    Form,
} from "@/components/ui/form";
import { Button } from "@/components/ui/button";
import { useForm } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { useState } from "react";
import { useImportUsers, useValidateAuthServers} from "@/components/administration/queries";
import { AuthServerUser, FormValueSignin, ImportUser} from "@/components/administration/types";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { toast } from "sonner";
import ImportUserStep1  from "./ImportUserStep1";
import ImportUserStep2 from "./ImportUserStep2";
import ImportUserStep3  from "./ImportUserStep3";
import { useQueryClient } from "@tanstack/react-query";

export default function ImportUserDialog({ children, onSuccess }: { children: React.ReactNode, onSuccess?: () => void }) {
    {/** General */}
    const { t } = useTranslation();
    const [open, setOpen] = useState(false);
    const [step, setStep] = useState<"sign-in" | "select-users" | "create-users">("sign-in");
    const [authServerInfo, setAuthServerInfo] = useState<FormValueSignin>();
    const queryClient = useQueryClient();

    {/** Sign in step */}
    const validateAuthServer = useValidateAuthServers();

    const formSignin = useForm<FormValueSignin>({
        defaultValues: {
            username: "",
            password: "",
            server: undefined,
        },
    });

    const onSubmitSignIn = (values: FormValueSignin) => {
        //TODO: CLIENT SIDE CREDS VALIDATION

        if (!values.username || !values.password || !values.server) {
            toast.error("Please enter a username, password and select a server");
            return;
        }

        validateAuthServer.mutate(values, {
            onSuccess: () => {
                setAuthServerInfo(values);
                setStep("select-users");
                formSignin.reset();
            }
        });
    }

    {/** Select users step */} 
    const [selectedUsers, setSelectedUsers] = useState<AuthServerUser[]>([]);           // Keeps track of the selected users for the next step

    {/** Create Users  */}
    const importUsers = useImportUsers();   
    const [formUsers, setFormUsers] = useState<ImportUser[]>([]);
    const [selectedIndices, setSelectedIndices] = useState<number[]>(formUsers.map((_, index) => index));

    const formImport = useForm<{ formUsers: ImportUser[] }>({
       values: { formUsers }
    });

    const { handleSubmit } = formImport;

    const onSubmit = (data: { formUsers: ImportUser[] }) => {

        const selectedFormUsers = selectedIndices.map(index => data.formUsers[index]); //REMOVES THE USERS THAT AREN'T SELECTED

        const incompleteUsers = selectedFormUsers.filter(user => !user.username?.trim() || !user.email?.trim()); //ESNURES ALL USERS ARE COMPLETE

        if (incompleteUsers.length > 0) {
            toast.error("Please fill in all required fields for selected users.");
            return;
        }

        //TODO:CLIENT SIDE VALIDATION with ZOD

        importUsers.mutate(selectedFormUsers, {
            onSuccess: () => {
                onSuccess?.();
                queryClient.invalidateQueries({ queryKey: ["users"] });
                setOpen(false);
                resetDialog();
            }
        });
    };
    
    const resetDialog = () => {
        setStep("sign-in");
        formSignin.reset();
        formImport.reset();
        setAuthServerInfo(undefined);
        setSelectedUsers([]);
        setFormUsers([]);
        setSelectedIndices([]);
    };

    
    return (
        <Dialog open={open} onOpenChange={(val) => {
            setOpen(val);
            if (!val) resetDialog();
        }}>
            <DialogTrigger asChild>
                {children}
            </DialogTrigger>
            
            {step === "sign-in" && (
                <DialogContent className="p-0 rounded-lg flex flex-col">
                    <DialogHeader className="bg-card rounded-t-lg border-b p-6 flex-shrink-0">
                        <DialogTitle className="text-xl font-semibold">
                            {t("LDAP / Active Directory Import Users")}
                            <span className="text-sm bg-muted rounded-sm m-5 py-1 px-2">{t("Step 1")}</span>
                        </DialogTitle>
                        <DialogDescription>
                            {t("Log into your LDAP / Active Directory administrator or privileged account to be able to import users directly into Network Analyzer.")}
                        </DialogDescription>
                    </DialogHeader>

                    <div className="flex-1">
                    <Form {...formSignin}>
                        <form onSubmit={formSignin.handleSubmit(onSubmitSignIn)}>
                            
                            <ImportUserStep1 form={formSignin} />

                            <DialogFooter className="bg-card rounded-b-lg border-t p-4 flex-shrink-0 mt-auto">
                                <Button type="button" variant="secondary" className="rounded-sm" onClick={() => setOpen(false)}>
                                    {t("Cancel")}
                                </Button>
                                <Button type="submit" className="bg-primary rounded-sm">
                                    {t("Next")}
                                </Button>
                            </DialogFooter>
                        </form>
                    </Form>
                    </div>
                </DialogContent>
            )}  

            {step === "select-users" && (
                <DialogContent className="sm:max-w-[75vw] rounded-lg z-50 flex flex-col p-0">
                    <DialogHeader className="bg-card rounded-t-lg border-b p-6">    
                        <DialogTitle>
                            {t("Select Users to Import")}
                            <span className="text-sm bg-muted rounded-sm m-5 py-1 px-2">{t("Step 2")}</span>
                        </DialogTitle>
                        <DialogDescription>
                            {t("Select the users you would like to give access to Network Analyzer via LDAP / Active Directory authentication. You will be able to set user-specific permissions on the next page.")}
                        </DialogDescription>
                    </DialogHeader>

                    {authServerInfo && (
                        <ImportUserStep2
                            authServerInfo={authServerInfo}
                            selectedUsers={selectedUsers}
                            setSelectedUsers={setSelectedUsers}
                        />
                    )}

                    <DialogFooter className="flex bg-card rounded-b-lg border-t p-4">
                        <Button variant="secondary" className="rounded-sm justify-left" onClick={() => setStep("sign-in")}>
                            {t("Back")}
                        </Button>
                        <Button
                            variant="default"
                            className="rounded-sm bg-primary justify-right ml-auto"
                            onClick={() => {
                                // Re-map form users from selected users when advancing to this step
                                const mappedUsers = selectedUsers.map((u) => ({
                                    username: u.username || "",
                                    email: u.email || "",
                                    displayname: u.displayname?.trim() ||
                                                    (u.givenname && u.sn ? `${u.givenname} ${u.sn}` : null) ||
                                                    (u.cn ? u.cn.split(" ").slice(0, 2).join(" ") : "") ||
                                                    "",
                                    role_id: 2,
                                    apiaccess: false,
                                    type: authServerInfo?.server.type?.toLowerCase() === "ldap" ? "LDAP" : "AD",
                                    auth_server_id: authServerInfo?.server.id,
                                    auth_server_data: u.dn,
                                }));
                                
                                setFormUsers(mappedUsers);
                                setSelectedIndices(mappedUsers.map((_, index) => index));
                                setStep("create-users");
                            }}
                            disabled={selectedUsers.length === 0}
                        >
                            {t("Add Selected")} ({selectedUsers.length})
                        </Button>
                    </DialogFooter>
                </DialogContent>
            )}

            {step === "create-users" && (
                <DialogContent className="sm:max-w-[80vw] h-[700px] rounded-lg flex flex-col p-0 gap-0">
                    <DialogHeader className="bg-card rounded-t-lg p-6 pb-0">
                        <DialogTitle>
                            {t("LDAP / Active Directory Import Users")}
                            <span className="text-sm bg-muted rounded-sm m-5 py-1 px-2">{t("Step 3")}</span>
                        </DialogTitle>
                        <DialogDescription>
                            {t("Review and confiugre settings for selected users. Complete all required fields before importing")}
                        </DialogDescription>
                    </DialogHeader>

                    <Form {...formImport}>
                        <form onSubmit={handleSubmit(onSubmit)} className="flex-1 flex flex-col">
                            <ImportUserStep3
                                formImport={formImport}
                                formUsers={formUsers}
                                selectedIndices={selectedIndices}
                                setSelectedIndices={setSelectedIndices}
                            />
                            
                            <DialogFooter className="flex bg-card rounded-b-lg border-t p-4">
                                <Button variant="secondary" onClick={() => setStep("select-users")}>
                                    {t("Back")}
                                </Button>
                                <Button type="submit">
                                    {t("Create Users")}
                                </Button>
                            </DialogFooter>
                        </form>
                    </Form>
                </DialogContent>
            )}
        </Dialog>
    );
}
