import React, { useEffect, useState } from 'react';
import { Button } from '@/components/ui/button';
import { ArrowDownAZ, ArrowDownZA, ChevronDown, ChevronRight, CircleMinus, CirclePlus, Folder, Search, User } from 'lucide-react';
import { AuthServerUser, FormValueSignin } from '@/components/administration/types';
import { Card, CardContent, CardHeader } from '../ui/card';
import {useQueryAuthServer} from '@/components/administration/queries';
import { useTranslation } from "react-i18next";
import { Input } from '../ui/input';
import { Checkbox } from '../ui/checkbox';
import { ScrollArea } from '../ui/scroll-area';
import { useQueryClient } from '@tanstack/react-query';
import { Select, SelectContent, SelectGroup, SelectItem, SelectTrigger, SelectValue } from '../ui/select';

const searchOptions = [
    { value: 'cn', label: 'Common Name' },
    { value: 'givenName', label: 'First Name'},
    { value: 'mail', label: 'Email' },
    { value: 'sn', label: 'Last Name'},
    { value: 'uid', label: 'UID' },
    { value: 'objectclass', label: 'Objectclass'},
    { value: 'samaccountname', label: 'sAMAccountName'},
    { value: 'displayname', label: 'Display Name'},
    { value: 'Custom', label: 'Custom'},
];

interface DirectoryProps {
    authServerInfo: FormValueSignin;
    selectedUsers: AuthServerUser[];
    setSelectedUsers: React.Dispatch<React.SetStateAction<AuthServerUser[]>>;
}

const arraysEqual = (a: string[], b: string[]): boolean => { //FUNCTION IS USED TO highlight a selected folder
    if (a.length !== b.length) return false;
    return a.every((item, index) => item === b[index]);
};

const getUsersFromPath = (json: Record<string, unknown>, path: string[]): AuthServerUser[] | null => { //RETREIVES UsersToShow from the json
    let current = json;
    for (const key of path) {
        if (current && typeof current === 'object' && key in current) {
        const next = current[key];
        if (typeof next === 'object' && next !== null) {
            current = next as Record<string, unknown>;
        } else {
            return null;
        }
        } else {
        return null;
        }
    }
    return current && 'UsersToShow' in current ? current['UsersToShow'] as AuthServerUser[] : null;
};

interface FolderItemProps {
    folderName: string; //Current folder
    folder: Record<string, unknown>;    //Object JSON for the folder
    path: string[];     //array of folder names to current folder location
    expandedFolders: Set<string>;   //Stringified paths to show which folders are open
    setExpandedFolders: (fn: (prev: Set<string>) => Set<string>) => void; 
    selectedPath: string[] | null;              //Current selected path
    setSelectedPath: (path: string[]) => void;
    setSearchQuery: (query: string) => void;
}

export function FolderItem({folderName, folder, path, expandedFolders, setExpandedFolders, selectedPath, setSelectedPath, setSearchQuery}: FolderItemProps) {
    const hasSubfolders = Object.keys(folder).some(     //used to check to see if a drop down needs to be added
        (key) => key !== 'UsersToShow' && typeof folder[key] === 'object'
    );

    const isExpanded = expandedFolders.has(path.join(' > ')); //Tracks whether a folder is expanded or not

    const toggleExpand = () => {
        setExpandedFolders((prev) => {
            const newSet = new Set(prev);
            const pathStr = path.join(' > ');
            if (newSet.has(pathStr)){
                newSet.delete(pathStr);
            } 
            else {
                newSet.add(pathStr);
            }
            return newSet;
        });
    };

    const handleSelect = () => {
        setSelectedPath(path);
        setSearchQuery('');
    };

    const subfolders = Object.entries(folder).filter(   //filters out UsersToShow and non objects
        ([key, value]) => key !== 'UsersToShow' && typeof value === 'object'
    );

    const userCount = Array.isArray(folder.UsersToShow) ? folder.UsersToShow.length : 0;

    return (
        <div className="ml-2 ">
            <div className="flex items-center">
                {hasSubfolders ? (
                    <Button variant="ghost" size="icon" 
                        onClick={toggleExpand} 
                    >
                        {isExpanded ? <ChevronDown size={8} /> : <ChevronRight size={8} />}
                    </Button>
                ): (
                    <span className="w-9" /> // Placeholder for alignment when no subfolders
                )}
                <Button
                    variant="link"
                    onClick={handleSelect}
                    className={`text-muted-foreground text-sm items-center ${
                    selectedPath && arraysEqual(selectedPath, path) ? 'text-primary' : ''}`}
                >
                <Folder size={16} />
                    {folderName}
                </Button>
                <span className={'text-right ml-auto text-sm bg-muted rounded-md p-1 px-2 text-muted-foreground'}>
                    {userCount}
                </span>
            </div>
            {isExpanded &&
                subfolders.map(([subFolderName, subFolder]) => (
                    <FolderItem
                        key={subFolderName}
                        folderName={subFolderName}
                        folder={subFolder as Record<string, unknown>}
                        path={[...path, subFolderName]}
                        expandedFolders={expandedFolders}
                        setExpandedFolders={setExpandedFolders}
                        selectedPath={selectedPath}
                        setSelectedPath={setSelectedPath}
                        setSearchQuery={setSearchQuery}
                    />
                ))
            }
        </div>
    );
};

export default function ImportUserStep2({authServerInfo, selectedUsers, setSelectedUsers}: DirectoryProps) {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const [searchInput, setSearchInput] = useState('');
    const [searchQuery, setSearchQuery] = useState<string | "">("");
    const [searchParam, setSearchParam] = useState<string | "">("cn");
    const [customParam, setCustomParam] = useState('');
    const { data: json, isLoading, error } = useQueryAuthServer(authServerInfo, searchInput, searchParam);

    const [expandedFolders, setExpandedFolders] = useState<Set<string>>(new Set());
    const [selectedPath, setSelectedPath] = useState<string[] | null>(null);
    const [expandedUsers, setExpandedUsers] = useState<Set<string>>(new Set());
    const [sortOrder, setSortOrder] = useState<'asc' | 'desc'>('asc');

    // Set initial folder state when json loads
    useEffect(() => {
        if (json) {
            const rootKey = Object.keys(json)[0] || '';
            setExpandedFolders(new Set([rootKey]));
            setSelectedPath([rootKey]);
        }
    }, [json]);

    if (isLoading) {
        return (
          <div className="flex items-center justify-center gap-2 h-[620px]">
            {t("Loading ")}<div className="animate-spin inline-block size-4 border-3 border-current border-t-transparent text-primary rounded-full" role="status" aria-label="loading"></div>
          </div>
        );
      }
    if (error) return <div>{t('Error: Failed to load directory data')}</div>;

    const folderUsers = selectedPath ? getUsersFromPath(json, selectedPath) : null;

    const handleSearch = () => {
        if(searchParam === 'Custom'){
            setSearchParam(customParam);
        }
        queryClient.invalidateQueries({ queryKey: ["tree"] });
    };


    const toggleUserSelection = (user: AuthServerUser) => {
        setSelectedUsers((prev) =>
        prev.some((u) => u.dn === user.dn) ? prev.filter((u) => u.dn !== user.dn) : [...prev, user]);
    };

    const toggleUserDetails = (dn: string) => {
        setExpandedUsers((prev) => {
            const newSet = new Set(prev);
            if (newSet.has(dn)) {
                newSet.delete(dn);
            } else {
                newSet.add(dn);
            }
            return newSet;
        });
    };

    return (
        <div className="px-6 max-h-[618px]">

            {/* BREADCRUMB CARD */}
            
            <Card className="p-2 mb-4">
                <CardContent className="p-0">
                    {selectedPath ? (
                        <nav className="flex flex-wrap items-center">
                            {selectedPath.map((folder, index) => {
                                const segmentPath = selectedPath.slice(0, index + 1);
                                return (
                                    <div key={index} className="flex items-center gap-1">
                                        {index > 0 && <span className="text-muted-foreground"> <ChevronRight size={16} /></span>}
                                        <Button
                                            variant="link"
                                            className={`text-sm p-2 ${
                                                index === selectedPath.length - 1 ? 'text-primary' : 'text-muted-foreground'
                                            } overflow-wrap-break-word`}
                                            onClick={() => setSelectedPath(segmentPath)}
                                            >
                                            {folder}
                                        </Button>
                                    </div>
                                );
                            })}
                            
                        </nav>
                    ) : (
                        <></>
                    )}
                </CardContent>
            </Card>

            <div className="flex gap-6">
                <Card className="pt-3 pb-0 rounded-b-none shrink-0 min-w-76">
                    <CardHeader className="text-lg font-semibold">
                        {t("Directory Tree")}
                        <div className="relative flex">
                            <div className="pr-2">
                                <Select
                                    value={searchParam}
                                    onValueChange={(value) => {
                                    setSearchParam(value);
                                    if (value !== 'Custom') setCustomParam(''); // reset custom input if not selected
                                    }}
                                >
                                    <SelectTrigger>
                                        <SelectValue/>
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectGroup>
                                            {searchOptions.map((option) => (
                                                <SelectItem key={option.value} value={option.value}>
                                                    {option.label}
                                                </SelectItem>
                                            ))}
                                        </SelectGroup>
                                    </SelectContent>
                                    
                                </Select>
                                    {searchParam === 'Custom' && (
                                        <Input
                                        className="mt-2 bg-background"
                                        placeholder="Enter custom field"
                                        value={customParam}
                                        onChange={(e) => setCustomParam(e.target.value)}
                                        />
                                    )}
                            </div>
                        
                            <Input
                                value={searchInput}
                                onChange={(e) => setSearchInput(e.target.value)}
                                onKeyDown={(e) => {
                                    if (e.key === 'Enter') {
                                        handleSearch();
                                    }
                                }}
                                placeholder={`${searchOptions.find(opt => opt.value === searchParam)?.label ?? '...'}`}
                                className="pr-8 bg-background"
                            />
                            <Button
                                variant="ghost"
                                onClick={handleSearch}
                                className="absolute inset-y-0 right-0 flex"
                            >
                                <Search size={16} className="text-muted-foreground" />
                            </Button>
                        </div>
                    </CardHeader>
                    <ScrollArea className = "bg-background">
                        <CardContent className="rounded-t-xl h-110">
                            {Object.entries(json).map(([folderName, folder]) => (
                                <FolderItem
                                    key={folderName}
                                    folderName={folderName}
                                    folder={folder as Record<string, unknown>}
                                    path={[folderName]}
                                    expandedFolders={expandedFolders}
                                    setExpandedFolders={setExpandedFolders}
                                    selectedPath={selectedPath}
                                    setSelectedPath={setSelectedPath}
                                    setSearchQuery={setSearchQuery}
                                />
                            ))}
                        </CardContent>
                    </ScrollArea>
                </Card>
                
                <Card className="pt-3 pb-0 rounded-t-xl rounded-b-none w-full">
                    <CardHeader className="text-lg font-semibold">
                        <div className="flex">
                             {selectedPath ? (
                                <>
                                    {t("Users in")} "{selectedPath.slice(-1)[0]}"
                                </>
                             ) : (
                                t("No folder selected")
                             )}
                            <div className="flex ml-auto gap-2">
                               {folderUsers && folderUsers.length > 0 && (
                                    <>
                                        <Button
                                            onClick={() => {
                                                setSelectedUsers([
                                                    ...selectedUsers,
                                                    ...folderUsers.filter(user => !selectedUsers.some(selected => selected.dn === user.dn))
                                                ]);
                                            }}
                                        >
                                            {t("Select All")}
                                        </Button>
                                        <Button
                                            variant="outline"
                                            onClick={() => {
                                                setSelectedUsers(selectedUsers.filter(selected => !folderUsers.some(user => user.dn === selected.dn)));
                                            }}
                                        >
                                            {t("Unselect All")}
                                        </Button>
                                    </>
                                )}
                                <Button
                                    variant="outline"
                                    size="icon"
                                    onClick={() => {
                                        setSortOrder(prev => {
                                            if (prev === 'asc') return 'desc';
                                            return 'asc';
                                        });
                                    }}
                                >
                                    {sortOrder === 'asc' ? <ArrowDownAZ size={16} /> : <ArrowDownZA size={16} />}
                                </Button>
                            </div>
                        </div>
                    </CardHeader>
                    <ScrollArea className="bg-background">
                        <CardContent className="rounded-b-none bg-background h-112 p-0 w-full">
                            
                            {folderUsers && folderUsers.length > 0 ? (
                                <div>
                                    {[...folderUsers]
                                    .sort((a, b) => {
                                        const usernameA = a.username || a.cn || '';
                                        const usernameB = b.username || b.cn || '';
                                        return searchQuery
                                            ? 0 
                                            : sortOrder === 'asc'
                                            ? usernameA.localeCompare(usernameB)
                                            : usernameB.localeCompare(usernameA);
                                    })
                                    .map((user) => (
                                        <div key={user.dn} className="border-b ">
                                            <div className="flex items-center pl-3 gap-2">
                                                <Checkbox
                                                    checked={selectedUsers.some((u) => u.dn === user.dn)}
                                                    onCheckedChange={() => toggleUserSelection(user)}
                                                />
                                                <User size={16} className="text-muted-foreground" />
                                                <span className="flex text-sm">
                                                    {user.displayname || user.username ||  user.givenname + user.sn || user.cn || 'Unknown'} 
                                                </span>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    onClick={() => toggleUserDetails(user.dn)}
                                                    className="ml-auto"
                                                >
                                                    {expandedUsers.has(user.dn) ? (
                                                    <CircleMinus size={12} />
                                                    ) : (
                                                    <CirclePlus size={12} />
                                                    )}
                                                </Button>
                                            </div>
                                            {expandedUsers.has(user.dn) && (
                                            <div className="pb-2 px-4">

                                                <div className="flex items-center text-muted-foreground">
                                                <div className="font-bold">{t("DN:")}</div>
                                                    <span className="text-right ml-auto overflow-hidden text-ellipsis whitespace-nowrap">{user.dn || 'N/A'}</span>
                                                </div>

                                                <div className="w-full my-2 border-b border-border" />

                                                <div className="flex items-center text-muted-foreground">
                                                    <div className="font-bold">{t("CN: ")}</div>
                                                    <span className="text-right ml-auto">{user.cn || 'N/A'}</span>
                                                </div>

                                                <div className="w-full my-2 border-b border-border" />

                                                <div className="flex items-center text-muted-foreground">
                                                <div className="font-bold">{t("Unique ID:")}</div>
                                                    <span className="text-right ml-auto">{user.username || 'N/A'}</span>
                                                </div>

                                                <div className="w-full my-2 border-b border-border" />

                                                <div className="flex items-center text-muted-foreground">
                                                    <div className="font-bold">{t("Email:")}</div>
                                                    <span className="text-right ml-auto p-1 rounded-md">{user.email || 'N/A'}</span>
                                                </div>

                                                <div className="w-full my-2 border-b border-border" />
                                                
                                                <div className="flex items-center text-muted-foreground">
                                                    <div className="font-bold">{t("Object Classes:")}</div>
                                                    <div className="flex flex-wrap ml-auto gap-2">
                                                        {user.oc ? (
                                                        Array.from({ length: user.oc.count }, (_, i) => (
                                                            <span key={i} className="px-2 py-1 bg-muted rounded-md text-center text-xs">
                                                                {user.oc[i.toString()] || 'N/A'}
                                                            </span> 
                                                        ))
                                                        ) : (
                                                        <span className="px-2 py-1 bg-muted-foreground rounded-md text-center text-sm">{user.oc ? Object.values(user.oc).filter(v => typeof v === 'string')[0] : 'N/A'}</span>
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                            )}
                                        </div>
                                    ))}
                                </div>
                                ) : (
                                <p className="flex justify-center items-center pt-10">{t("No users found")}</p>
                            )}
                            
                        </CardContent>
                    </ScrollArea>
                </Card>
            </div>
        </div>
    );
}