import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useTranslation } from "react-i18next";
import { useForm, SubmitHandler } from "react-hook-form";
import { Form, FormLabel, FormControl, FormField, FormItem, FormMessage } from "@/components/ui/form";
import { Card, CardContent } from "@/components/ui/card";
import { PageSeparator } from "@/components/Page";
import { useActivateLicense, useGetLicenseKey } from "@/components/administration/license/queries";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { Info, Loader2 } from "lucide-react"; 
import { useAuth } from "@/contexts/AuthContext";
import { Badge } from "@/components/ui/badge";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { obfuscateKey } from "@/components/administration/license/utils";
import { useState } from "react";

interface ActivateLicenseFormProps {
  showBadge?: boolean;
  enableLicenseKey?: boolean;
}

export default function ActivateLicenseForm({ 
  showBadge = true,
  enableLicenseKey = false,
}: ActivateLicenseFormProps) {
  const { t } = useTranslation();
  const activateLicense = useActivateLicense();
  const { data: licenseKey } = useGetLicenseKey();
  const { license } = useAuth();
  const isActivating = activateLicense.isPending;

  const originalKey = licenseKey ?? '';
  const [hasUserModified, setHasUserModified] = useState(false);

  const activateLicenseSchema = z.object({
    clientId: z.string().min(1, t("Client ID is required")),
    key: z.string().min(1, t("License key is required")),
 });

  type ActivateLicenseFormData = z.infer<typeof activateLicenseSchema>;

  const form = useForm<ActivateLicenseFormData>({
    resolver: zodResolver(activateLicenseSchema),
    values: {
      clientId: "",
      key: originalKey,
    }
  });

  const { isDirty } = form.formState;
  const currentKeyValue = form.watch('key') ?? '';

  const onSubmit: SubmitHandler<ActivateLicenseFormData> = (data) => {
    activateLicense.mutate({
      clientId: data.clientId ?? '',
      licenseKey: data.key,
    });
  };

  return (
    <Card>
      <CardContent>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <div className="flex flex-row items-center justify-between">
              <h1 className="leading-9 text-md font-large font-semibold">
                {t("License Activation")}
                {showBadge && (
                  <Badge
                    className="ml-2"
                    variant={license?.activated ? "green" : "red"}
                  >
                    {t(license?.activated ? "Activated" : "Not Activated")}
                  </Badge>
                )}
              </h1>
              <Button
                disabled={!isDirty || isActivating}
                variant={isDirty ? "default" : "outline"}
                className={`${isActivating ? ' opacity-80' : ''}`}
                type="submit"
              >
                {isActivating ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    {t("Activating...")}
                  </>
                ) : (
                  t("Activate License")
                )}
              </Button>
            </div>

            <PageSeparator />

            <FormField
              control={form.control}
              name="clientId"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <div className="flex items-center gap-2">
                    <FormLabel>{t("Client ID / Token")}</FormLabel>
                    <TooltipProvider>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <span tabIndex={0} className="cursor-pointer">
                            <Info size={16} />
                          </span>
                        </TooltipTrigger>
                        <TooltipContent className="max-w-80 text-wrap">
                          {t(
                            "Your client ID or token should be in the license email. Contact sales@nagios.com if you do not know yours."
                          )}
                        </TooltipContent>
                      </Tooltip>
                    </TooltipProvider>
                  </div>
                  <div className="w-md">
                    <FormControl>
                      <Input
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </div>
                </FormItem>
              )}
            />

            <PageSeparator />

            <FormField
              control={form.control}
              name="key"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("License Key")}</FormLabel>
                  <div className="w-md">
                    <FormControl>
                      <Input
                        {...field}
                        value={enableLicenseKey ? currentKeyValue : (hasUserModified ? currentKeyValue : obfuscateKey(originalKey))}
                        onChange={(e) => {
                          const newValue = e.target.value;
                          field.onChange(newValue);
                          setHasUserModified(newValue !== originalKey);
                        }}
                      />
                    </FormControl>
                    <FormMessage />
                  </div>
                </FormItem>
              )}
            />
          </form>
        </Form>
      </CardContent>
    </Card>
  );
}
