import { Button } from "@/components/ui/button";
import { useTranslation } from "react-i18next";
import { Card, CardContent } from "@/components/ui/card";
import { PageSeparator } from "@/components/Page";
import { useSetLicenseType, useExtendTrial } from "@/components/administration/license/queries";
import type { License } from "@/components/administration/license/types";
import { useAuth } from "@/contexts/AuthContext";
import { Form, FormField, FormItem, FormLabel, FormControl } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { useForm, SubmitHandler } from "react-hook-form";
import { Badge } from "@/components/ui/badge";

interface TrialFormProps {
  showBorder?: boolean;
}

export default function TrialForm({ showBorder = false }: TrialFormProps) {
  const { t } = useTranslation();
  const setLicenseType = useSetLicenseType();
  const extendTrial = useExtendTrial();
  const { license } = useAuth();

  const form = useForm<{ trial_extension_key: string }>({
    defaultValues: {
      trial_extension_key: "",
    },
  });
  
  const { isDirty } = form.formState;

  const updateLicenseType = (type: License['type']) => {
    if (license?.type === "trial" || license?.trial_expired) return;
    setLicenseType.mutate({ type });
  };

  const onSubmit: SubmitHandler<{ trial_extension_key: string }> = (data) => {
    extendTrial.mutate(data.trial_extension_key);
  };

  return (
    <Card
      className={` 
        ${!license?.trial_expired ? " cursor-pointer" : ""} 
        ${showBorder && license?.type === "trial" ? " border-badge-green-border" : ""}
      `}
      onMouseDown={() => {
        updateLicenseType("trial");
      }}
    >
      <CardContent>
        <div className="flex flex-row items-center justify-between">
          <h1 className="leading-9 text-md font-large font-semibold">
            {t("Trial")}
            {license && !license.trial_expired && (
              <Badge
                className="ml-2"
                variant={
                  license.trial_days_left >= 20
                    ? "green"
                    : license.trial_days_left >= 10
                    ? "yellow"
                    : "red"
                }
              >
                {license.trial_days_left} {t("Days Left")}
              </Badge>
            )}
          </h1>
          <Button
            disabled={!isDirty}
            variant={isDirty ? "default" : "outline"}
            className="border"
            type="submit"
            form="trial-extension-form"
          >
            {t("Extend Trial")}
          </Button>
        </div>

        <PageSeparator />

        <div onMouseDown={(e) => e.stopPropagation()}>
          <Form {...form}>
            <form id="trial-extension-form" onSubmit={form.handleSubmit(onSubmit)}>
              <FormField
                control={form.control}
                name="trial_extension_key"
                render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Trial Extension Key")}</FormLabel>
                  <FormControl className="w-md">
                    <Input {...field} />
                  </FormControl>
                </FormItem>
                )}
              />
            </form>
          </Form>
        </div>
      </CardContent>
    </Card>
  );
}

