import { useMutation, useQueryClient } from "@tanstack/react-query";
import axios, { AxiosError } from "@/lib/axios";
import { useTranslation } from "react-i18next";
import { toast } from "sonner";
import type { License } from "@/components/administration/license/types";
import { useTranslateError } from "@/components/administration/license/utils";
import { useQuery } from "@tanstack/react-query";

type ErrorResponse = {
  success?: boolean;
  title?: string;
  message?: string;
}

export function useGetLicenseKey() {
  return useQuery<string>({
    queryKey: ["licenseKey"],
    queryFn: async () => axios.get("/api/v1/license/key").then(res => res.data?.key ?? ''),
  });
}

export function useSetLicenseKey() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (key: string) =>
      axios.put("/api/v1/license/key", {
        key
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["license"] });
      queryClient.invalidateQueries({ queryKey: ["licenseKey"] });
      toast(t("License key successfully updated"));
    },
    onError: (error: AxiosError) => {
      if (error?.response?.status === 422) {
        toast(t("License key is invalid."));
      } else {
        toast(t("An error occurred while updating the license key."));
      }
    },
  });
}

export function useExtendTrial() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (trial_extension_key: string) =>
      axios.put("/api/v1/license/trial/extend", {
        trial_extension_key
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["license"] });
      toast(t("Trial successfully extended"));
    },
    onError: (error: AxiosError) => {
      if (error?.response?.status === 422) {
        toast(t("Trial extension key is invalid."));
      } else {
        toast(t("An error occurred while extending the trial."));
      }
    },
  })
}

export function useSetLicenseType() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation({
    mutationFn: (data: Pick<License, "type">) =>
      axios.patch("/api/v1/license/type", data),
    onError: (error: AxiosError) => {
      if (error?.response?.status === 500) {
        toast(t("An error occurred while updating the license type."));
      }
    },
    onSuccess: () => {
      toast(t("License type updated"));
      queryClient.invalidateQueries({ queryKey: ["license"] });
    },
  });
}

export function useActivateLicense() {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  const translateError = useTranslateError();

  return useMutation({
    mutationFn: (data: { clientId: string; licenseKey: string }) =>
      axios.post("/api/v1/license/activate", data),
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ["license"] });
      toast(response.data.message || t("License activated successfully"));
    },
    onError: (error: AxiosError<ErrorResponse>) => {
      if (error.response?.data?.title && error.response?.data?.message) {
        const { title, message } = translateError(error.response.data.title, error.response.data.message);
        toast.error(
          title,
          {
            description: message,
            duration: 30000, // 30 seconds
            closeButton: true,
          }
        );
      } else if (error.response?.data?.message && error.response?.data?.message == "Invalid license key.") {
        toast.error(t("Invalid license key."));
      } else {
        toast.error(t("An error occurred while activating the license."));
      }
    },
  });
}
